const { ipcRenderer } = require('electron');
const path = require('path');
const logger = require(path.join(__dirname, 'renderer', 'scripts', 'logger'));
const state = require(path.join(__dirname, 'renderer', 'scripts', 'state'));
const analyticsClient = require(path.join(__dirname, 'renderer', 'scripts', 'analytics', 'posthog'));
const { normalizeFilePath, formatFileSize, formatDate, formatTimestamp } = require(path.join(__dirname, 'renderer', 'scripts', 'utils'));
let videos = state.videos;
let videoMap = state.videoMap; // path -> video object
let currentVideo = state.currentVideo;
let viewMode = state.viewMode; // 'gallery', 'player', or 'editor'
let editorState = state.editorState; // Holds state for video editing workflow
let settings = state.settings;
let searchQuery = state.searchQuery;
const treeExpandedState = state.treeExpandedState;
let currentSelectedDirectory = state.currentSelectedDirectory;
let currentTreeSelection = state.currentTreeSelection || null;
state.videos = videos;
state.videoMap = videoMap;
state.currentVideo = currentVideo;
state.viewMode = viewMode;
state.editorState = editorState;
state.settings = settings;
state.searchQuery = searchQuery;
state.treeExpandedState = treeExpandedState;
state.currentSelectedDirectory = currentSelectedDirectory;
state.currentTreeSelection = currentTreeSelection;
if (!currentTreeSelection) {
    currentTreeSelection = '__all__';
    state.currentTreeSelection = currentTreeSelection;
}

// Frame-by-frame navigation state
let frameSkipInterval = null;
let frameSkipDirection = null; // 'left' or 'right'
let frameSkipTimeout = null; // For delayed start of continuous skipping
const FRAME_SKIP_DELAY_MS = 1000; // Wait 1 second before continuous skipping
const FRAME_SKIP_INTERVAL_MS = 200; // 5 fps = 200ms per frame
const SINGLE_FRAME_DURATION = 1 / 30; // Default 30fps for single frame skip
const youtubeAuthState = state.youtubeAuthState || {
    status: { authenticated: false },
    inProgress: false,
    manualAuthUrl: null,
    lastError: null
};
state.youtubeAuthState = youtubeAuthState;
const youtubeState = state.youtube || {
    accounts: [],
    activeAccountId: null,
    libraryByAccount: new Map(),
    libraryIndexByAccount: new Map(),
    videos: new Map(),
    jobs: new Map(),
    jobsByVideoPath: new Map(),
    lastGalleryRefresh: 0,
    library: createEmptyYouTubeLibrary()
};
if (!Array.isArray(youtubeState.accounts)) {
    youtubeState.accounts = [];
}
if (typeof youtubeState.activeAccountId !== 'string' && youtubeState.activeAccountId !== null) {
    youtubeState.activeAccountId = null;
}
if (!(youtubeState.libraryByAccount instanceof Map)) {
    youtubeState.libraryByAccount = new Map();
}
if (!(youtubeState.libraryIndexByAccount instanceof Map)) {
    youtubeState.libraryIndexByAccount = new Map();
}
if (!(youtubeState.videos instanceof Map)) {
    youtubeState.videos = new Map();
}
if (!(youtubeState.jobs instanceof Map)) {
    youtubeState.jobs = new Map();
}
if (!(youtubeState.jobsByVideoPath instanceof Map)) {
    youtubeState.jobsByVideoPath = new Map();
}
if (youtubeState.jobsByVideoPath instanceof Map) {
    const migratedJobsByVideoPath = new Map();
    youtubeState.jobsByVideoPath.forEach((value, key) => {
        if (typeof key === 'string' && key.includes('::')) {
            migratedJobsByVideoPath.set(key, value);
        } else {
            migratedJobsByVideoPath.set(buildYouTubeJobKey(key, null), value);
        }
    });
    youtubeState.jobsByVideoPath = migratedJobsByVideoPath;
}
if (typeof youtubeState.lastGalleryRefresh !== 'number') {
    youtubeState.lastGalleryRefresh = 0;
}
if (!youtubeState.library || typeof youtubeState.library !== 'object') {
    youtubeState.library = createEmptyYouTubeLibrary();
}
state.youtube = youtubeState;

let youtubeAccounts = youtubeState.accounts;
let youtubeActiveAccountId = youtubeState.activeAccountId || null;
const youtubeLibraryByAccount = youtubeState.libraryByAccount;
const youtubeLibraryIndexByAccount = youtubeState.libraryIndexByAccount;
const youtubeLibraryLoadingAccounts = state.youtubeLibraryLoadingAccounts instanceof Map ? state.youtubeLibraryLoadingAccounts : new Map();
state.youtubeLibraryLoadingAccounts = youtubeLibraryLoadingAccounts;
// Track accounts that are loading from cache (vs loading from YouTube)
const youtubeLibraryCacheLoadingAccounts = new Map();

let youtubeLibrary = null;
if (youtubeActiveAccountId && youtubeLibraryByAccount.has(youtubeActiveAccountId)) {
    youtubeLibrary = youtubeLibraryByAccount.get(youtubeActiveAccountId);
} else if (youtubeLibraryByAccount.size > 0) {
    const firstEntry = youtubeLibraryByAccount.entries().next();
    if (!firstEntry.done) {
        youtubeActiveAccountId = firstEntry.value[0];
        youtubeState.activeAccountId = youtubeActiveAccountId;
        youtubeLibrary = firstEntry.value[1];
    }
}
if (!youtubeLibrary) {
    youtubeLibrary = { ...createEmptyYouTubeLibrary(), ...youtubeState.library };
    if (youtubeActiveAccountId) {
        youtubeLibrary.accountId = youtubeActiveAccountId;
        youtubeLibraryByAccount.set(youtubeActiveAccountId, youtubeLibrary);
    }
}
state.youtube.accounts = youtubeAccounts;
state.youtube.activeAccountId = youtubeActiveAccountId;
state.youtube.library = youtubeLibrary;
state.youtube.libraryByAccount = youtubeLibraryByAccount;
state.youtube.libraryIndexByAccount = youtubeLibraryIndexByAccount;

let youtubeLibraryVideoIndex = youtubeLibraryIndexByAccount.get(youtubeLibrary.accountId || '__default__') || null;
let currentYouTubeAccountId = state.currentYouTubeAccountId || youtubeActiveAccountId || null;
state.currentYouTubeAccountId = currentYouTubeAccountId;
let currentYouTubeCategory = state.currentYouTubeCategory || null;
let currentYouTubePlaylist = state.currentYouTubePlaylist || null;
state.currentYouTubeCategory = currentYouTubeCategory;
state.currentYouTubePlaylist = currentYouTubePlaylist;
// Initialize loading state to false - will be set correctly when loading actually happens
let youtubeLibraryLoading = false;
const YOUTUBE_ROOT_KEY = '__youtube__';
const YOUTUBE_ACCOUNT_PREFIX = '__youtube_account__';
const YOUTUBE_CATEGORY_PREFIX = '__youtube_category__';
const YOUTUBE_PLAYLIST_PREFIX = '__youtube_playlist__';
const YOUTUBE_CATEGORIES = [
    { key: 'videos', label: 'Videos' },
    { key: 'shorts', label: 'Shorts' },
    { key: 'live', label: 'Live' }
];
const LOCAL_ROOT_KEY = '__all__';
const YOUTUBE_PRIVACY_OPTIONS = ['public', 'unlisted', 'private'];
const YOUTUBE_LIBRARY_CACHE_MS = 60 * 1000;

let lastTrackedScreen = viewMode || 'gallery';
let searchAnalyticsTimer = null;
let lastTrackedSearchQuery = '';

function clamp(value, min, max) {
    return Math.min(Math.max(value, min), max);
}

function getBaseAnalyticsProps(additional = {}) {
    return {
        view_mode: viewMode,
        total_videos: videos.length,
        monitored_directory_count: Array.isArray(settings.monitoredDirectories) ? settings.monitoredDirectories.length : 0,
        youtube_connected: youtubeAuthState && youtubeAuthState.status && youtubeAuthState.status.authenticated ? 1 : 0,
        ...additional
    };
}

function summarizeDirectoryPath(dirPath) {
    if (!dirPath || typeof dirPath !== 'string') {
        return {
            directory_depth: 0,
            has_spaces: false,
            is_network_path: false
        };
    }
    const normalized = dirPath.replace(/\\/g, '/');
    const segments = normalized.split('/').filter(Boolean);
    return {
        directory_depth: segments.length,
        has_spaces: /\s/.test(dirPath) ? 1 : 0,
        is_network_path: normalized.startsWith('\\\\') || normalized.startsWith('//') ? 1 : 0
    };
}

function getVideoAnalyticsProps(video) {
    if (!video || typeof video !== 'object') {
        return {};
    }
    const extension = typeof video.extension === 'string' && video.extension.length > 0
        ? video.extension.toLowerCase()
        : 'unknown';
    const sizeMb = Number.isFinite(video.size) ? clamp(Math.round(video.size / (1024 * 1024)), 0, 50000) : null;
    const directoryProps = summarizeDirectoryPath(video.directoryPath || video.directory || '');
    return {
        extension,
        size_mb: sizeMb,
        is_transcode_candidate: needsLiveTranscode(video) ? 1 : 0,
        has_youtube_record: youtubeState.videos && youtubeState.videos.has(video.path) ? 1 : 0,
        ...directoryProps
    };
}

function trackEvent(eventName, properties = {}, options = {}) {
    if (!eventName) {
        return;
    }
    const payload = getBaseAnalyticsProps(properties);
    analyticsClient.capture(eventName, payload, options).catch(() => {});
}

function trackScreenView(screenName, properties = {}) {
    if (!screenName) {
        return;
    }
    if (screenName === lastTrackedScreen && !properties.force) {
        return;
    }
    lastTrackedScreen = screenName;
    trackEvent('screen_view', {
        screen_name: screenName,
        ...properties
    });
}

function createEmptyYouTubeLibrary() {
    return {
        fetchedAt: 0,
        accountId: null,
        channelId: null,
        channelTitle: null,
        videos: [],
        playlists: [],
        playlistItems: {},
        errors: null
    };
}

function encodeKeyComponent(value) {
    if (value == null) {
        return '';
    }
    return encodeURIComponent(String(value));
}

function decodeKeyComponent(value) {
    if (value == null) {
        return null;
    }
    try {
        return decodeURIComponent(value);
    } catch (error) {
        return value;
    }
}

function buildYouTubeAccountKey(accountId) {
    return `${YOUTUBE_ACCOUNT_PREFIX}${encodeKeyComponent(accountId || '')}`;
}

function buildYouTubeCategoryKey(accountId, categoryKey) {
    return `${YOUTUBE_CATEGORY_PREFIX}${encodeKeyComponent(accountId || '')}__${encodeKeyComponent(categoryKey || '')}`;
}

function buildYouTubePlaylistKey(accountId, playlistId) {
    return `${YOUTUBE_PLAYLIST_PREFIX}${encodeKeyComponent(accountId || '')}__${encodeKeyComponent(playlistId || '')}`;
}

function buildYouTubeJobKey(videoPath, accountId) {
    return `${videoPath || ''}::${accountId || 'default'}`;
}

function getYouTubeAccountLabel(account) {
    if (!account) {
        return 'YouTube Account';
    }
    return account.channelTitle
        || account.channelCustomUrl
        || account.email
        || account.id
        || 'YouTube Account';
}

function ensureYouTubeLibraryCache(accountId) {
    if (!accountId) {
        return createEmptyYouTubeLibrary();
    }
    if (!youtubeLibraryByAccount.has(accountId)) {
        const snapshot = createEmptyYouTubeLibrary();
        snapshot.accountId = accountId;
        youtubeLibraryByAccount.set(accountId, snapshot);
    }
    return youtubeLibraryByAccount.get(accountId);
}

function setCurrentYouTubeAccount(accountId) {
    const previousAccountId = currentYouTubeAccountId;
    
    if (youtubeAccounts.length === 0) {
        currentYouTubeAccountId = null;
        state.currentYouTubeAccountId = currentYouTubeAccountId;
        youtubeActiveAccountId = null;
        state.youtube.activeAccountId = null;
        youtubeLibrary = createEmptyYouTubeLibrary();
        youtubeLibraryVideoIndex = null;
        youtubeLibraryLoading = false;
        state.youtube.library = youtubeLibrary;
        return;
    }

    if (!accountId) {
        accountId = youtubeActiveAccountId || (youtubeAccounts[0] && youtubeAccounts[0].id) || null;
    }

    currentYouTubeAccountId = accountId;
    state.currentYouTubeAccountId = currentYouTubeAccountId;
    if (accountId) {
        youtubeActiveAccountId = accountId;
        state.youtube.activeAccountId = accountId;
        youtubeLibrary = ensureYouTubeLibraryCache(accountId);
        if (!youtubeLibrary.accountId) {
            youtubeLibrary.accountId = accountId;
        }
        youtubeLibraryVideoIndex = youtubeLibraryIndexByAccount.get(accountId) || null;
        youtubeLibraryLoading = Boolean(youtubeLibraryLoadingAccounts.get(accountId));
        state.youtube.library = youtubeLibrary;
    } else {
        youtubeLibrary = createEmptyYouTubeLibrary();
        youtubeLibraryVideoIndex = null;
        youtubeLibraryLoading = false;
        state.youtube.library = youtubeLibrary;
    }

    syncUploadContextAccount(currentYouTubeAccountId);
    renderYouTubeAccountSwitcher();
    
    // Reload YouTube records when account changes (to get records filtered by new account)
    // Don't force refresh - check cache first to avoid unnecessary API calls
    if (previousAccountId !== currentYouTubeAccountId) {
        hydrateYouTubeState().catch(error => {
            logger.warn('Failed to reload YouTube records after account change:', error);
        });
    }
}

function getResolvedYouTubeAccountId(preferredAccountId = null) {
    if (preferredAccountId && typeof preferredAccountId === 'string') {
        return preferredAccountId;
    }
    if (currentYouTubeAccountId) {
        return currentYouTubeAccountId;
    }
    if (youtubeActiveAccountId) {
        return youtubeActiveAccountId;
    }
    if (Array.isArray(youtubeAccounts) && youtubeAccounts.length > 0) {
        const fallback = youtubeAccounts.find(account => account && account.id);
        return fallback ? fallback.id : null;
    }
    return null;
}

function getActiveYouTubeAccountRecord(accountId = getResolvedYouTubeAccountId()) {
    if (!accountId) {
        return { accountId: null, account: null };
    }
    const account = Array.isArray(youtubeAccounts)
        ? youtubeAccounts.find(entry => entry && entry.id === accountId)
        : null;
    return { accountId, account: account || null };
}

function syncUploadContextAccount(accountId = null) {
    const resolvedAccountId = getResolvedYouTubeAccountId(accountId);
    if (youtubeUploadContext) {
        youtubeUploadContext.accountId = resolvedAccountId;
    }
    return resolvedAccountId;
}

function renderYouTubeAccountSwitcher() {
    if (!youtubeAccountSwitcher) {
        return;
    }

    const isAuthenticated = Boolean(youtubeAuthState && youtubeAuthState.status && youtubeAuthState.status.authenticated);
    const accountList = Array.isArray(youtubeAccounts) ? youtubeAccounts.filter(account => account && account.id) : [];

    // If not authenticated or no accounts, hide everything
    if (!isAuthenticated || accountList.length === 0) {
        youtubeAccountSwitcher.classList.remove('active');
        youtubeAccountSwitcher.innerHTML = '';
        return;
    }

    youtubeAccountSwitcher.classList.add('active');
    youtubeAccountSwitcher.innerHTML = '';

    const activeAccountId = getResolvedYouTubeAccountId();
    const hasMultipleAccounts = accountList.length > 1;

    // Only show account switcher UI (label + list) if there are multiple accounts
    if (hasMultipleAccounts) {
        const label = document.createElement('span');
        label.className = 'youtube-account-switcher-label';
        label.textContent = 'Active account';
        youtubeAccountSwitcher.appendChild(label);

        const list = document.createElement('div');
        list.className = 'youtube-account-switcher-list';
        youtubeAccountSwitcher.appendChild(list);

        accountList.forEach((account) => {
            const button = document.createElement('button');
            button.type = 'button';
            button.className = `youtube-account-switcher-chip${account.id === activeAccountId ? ' active' : ''}`;
            button.dataset.accountId = account.id;
            button.title = getYouTubeAccountLabel(account);
            button.setAttribute('aria-pressed', account.id === activeAccountId ? 'true' : 'false');
            button.setAttribute('aria-label', `Switch to ${getYouTubeAccountLabel(account)}`);

            const avatarUrl = account.channelThumbnailUrl;
            if (avatarUrl) {
                const img = document.createElement('img');
                img.src = avatarUrl;
                img.alt = `${getYouTubeAccountLabel(account)} avatar`;
                button.appendChild(img);
            } else {
                const fallback = document.createElement('span');
                const labelText = getYouTubeAccountLabel(account) || 'YouTube';
                const initial = (labelText.trim().charAt(0).toUpperCase()) || 'Y';
                fallback.textContent = initial;
                button.appendChild(fallback);
            }

            button.addEventListener('click', () => {
                handleYouTubeAccountSwitch(account.id);
            });

            list.appendChild(button);
        });
    }

    // Refresh button is now in the sidebar, so we don't add it here anymore
}

function handleYouTubeAccountSwitch(accountId) {
    if (!accountId || accountId === currentYouTubeAccountId) {
        return;
    }

    setCurrentYouTubeAccount(accountId);
    const resolvedAccountId = syncUploadContextAccount(accountId);

    if (isYouTubeTreeKey(currentTreeSelection)) {
        const context = parseTreeSelection(currentTreeSelection);
        let nextSelection = null;
        if (context.type === 'playlist' && context.playlistId) {
            nextSelection = buildYouTubePlaylistKey(resolvedAccountId, context.playlistId);
        } else if (context.category) {
            nextSelection = buildYouTubeCategoryKey(resolvedAccountId, context.category);
        } else {
            const fallbackCategory = currentYouTubeCategory || 'videos';
            nextSelection = buildYouTubeCategoryKey(resolvedAccountId, fallbackCategory);
        }
        filterAndRenderGallery(nextSelection);
    } else {
        filterAndRenderGallery(currentTreeSelection);
    }

    updateDirectoryTree();

    if (resolvedAccountId) {
        // Load library for the switched account - will use cache if available
        loadYouTubeLibraryForAccount(resolvedAccountId, {}).catch((error) => {
            logger.warn('Failed to load YouTube library for account switch:', error && error.message ? error.message : error);
        });
    }
}

async function loadYouTubeLibraryForAccount(accountId, options = {}) {
    if (!ipcRenderer || !accountId) {
        return ensureYouTubeLibraryCache(accountId);
    }

    const cacheMs = Number.isFinite(options.cacheMs) ? options.cacheMs : YOUTUBE_LIBRARY_CACHE_MS;
    const force = options.force === true;
    const existingLibrary = youtubeLibraryByAccount.get(accountId);
    const now = Date.now();
    
    // Prevent multiple simultaneous loads for the same account
    const isAlreadyLoading = youtubeLibraryLoadingAccounts.get(accountId) || youtubeLibraryCacheLoadingAccounts.get(accountId);
    if (isAlreadyLoading && !force) {
        // Wait a bit and check if it's still loading, then return existing library if available
        await new Promise(resolve => setTimeout(resolve, 100));
        const stillLoading = youtubeLibraryLoadingAccounts.get(accountId) || youtubeLibraryCacheLoadingAccounts.get(accountId);
        if (stillLoading && existingLibrary) {
            return existingLibrary;
        }
    }
    
    // Check in-memory cache first
    if (!force && existingLibrary && existingLibrary.fetchedAt && now - existingLibrary.fetchedAt < cacheMs) {
        // Ensure loading state is false
        youtubeLibraryLoadingAccounts.set(accountId, false);
        youtubeLibraryCacheLoadingAccounts.set(accountId, false);
        if (accountId === currentYouTubeAccountId) {
            youtubeLibraryLoading = false;
        }
        return existingLibrary;
    }

    // If not in memory or expired, try loading from disk cache BEFORE setting loading state
    if (!force && (!existingLibrary || !existingLibrary.fetchedAt || now - existingLibrary.fetchedAt >= cacheMs)) {
        // Set cache loading flag (don't trigger gallery update here to avoid loops)
        youtubeLibraryCacheLoadingAccounts.set(accountId, true);
        
        try {
            const cacheResponse = await ipcRenderer.invoke('youtube.cache.load', { accountId });
            if (cacheResponse && cacheResponse.success && cacheResponse.cache) {
                const cachedLibrary = ensureYouTubeLibraryCache(accountId);
                cachedLibrary.accountId = cacheResponse.cache.accountId || accountId;
                cachedLibrary.channelId = cacheResponse.cache.channelId || null;
                cachedLibrary.channelTitle = cacheResponse.cache.channelTitle || null;
                cachedLibrary.videos = Array.isArray(cacheResponse.cache.videos) ? cacheResponse.cache.videos : [];
                cachedLibrary.playlists = Array.isArray(cacheResponse.cache.playlists) ? cacheResponse.cache.playlists : [];
                cachedLibrary.playlistItems = cacheResponse.cache.playlistItems && typeof cacheResponse.cache.playlistItems === 'object' ? cacheResponse.cache.playlistItems : {};
                cachedLibrary.fetchedAt = cacheResponse.cache.fetchedAt ? new Date(cacheResponse.cache.fetchedAt).getTime() : Date.now();
                
                youtubeLibraryByAccount.set(accountId, cachedLibrary);
                youtubeState.libraryByAccount = youtubeLibraryByAccount;
                state.youtube.libraryByAccount = youtubeLibraryByAccount;
                
                if (accountId === currentYouTubeAccountId || accountId === youtubeActiveAccountId) {
                    youtubeLibrary = cachedLibrary;
                    youtubeState.library = youtubeLibrary;
                    state.youtube.library = youtubeLibrary;
                    resetYouTubeLibraryIndex(accountId);
                }
                
                // Use cache forever - only fetch from YouTube if explicitly forced (refresh button)
                // This prevents unnecessary API calls and quota depletion
                // User can manually refresh if they make changes outside the app
                youtubeLibraryLoadingAccounts.set(accountId, false);
                youtubeLibraryCacheLoadingAccounts.set(accountId, false);
                if (accountId === currentYouTubeAccountId) {
                    youtubeLibraryLoading = false;
                    state.youtube.library = youtubeLibrary;
                    youtubeState.library = youtubeLibrary;
                }
                
                // Don't trigger gallery update here - let the natural flow handle it to avoid loops
                // The gallery will update when the caller processes the returned library
                
                // Only fetch from YouTube if explicitly forced (user clicked refresh button)
                if (force) {
                    // Force refresh requested, will fetch below
                } else {
                    // Use the cached library - cache is valid forever
                    return cachedLibrary;
                }
            }
        } catch (error) {
            logger.warn('Failed to load YouTube cache from disk:', error && error.message ? error.message : error);
        } finally {
            // Clear cache loading flag
            youtubeLibraryCacheLoadingAccounts.set(accountId, false);
        }
    }

    // Only fetch from YouTube if explicitly forced (user clicked refresh)
    // If cache wasn't found and force is false, return empty library instead of calling API
    if (!force) {
        logger.log('[Renderer] loadYouTubeLibraryForAccount: No cache found and force=false, returning empty library to avoid API call');
        const emptyLibrary = ensureYouTubeLibraryCache(accountId);
        emptyLibrary.videos = [];
        emptyLibrary.playlists = [];
        emptyLibrary.playlistItems = {};
        emptyLibrary.fetchedAt = Date.now();
        youtubeLibraryByAccount.set(accountId, emptyLibrary);
        youtubeLibraryLoadingAccounts.set(accountId, false);
        youtubeLibraryCacheLoadingAccounts.set(accountId, false);
        if (accountId === currentYouTubeAccountId) {
            youtubeLibraryLoading = false;
        }
        return emptyLibrary;
    }

    // Only set loading state to true if we actually need to fetch from YouTube (force is true)
    logger.log('[Renderer] loadYouTubeLibraryForAccount: Force=true, will call YouTube API for account:', accountId);
    youtubeLibraryLoadingAccounts.set(accountId, true);
    if (accountId === currentYouTubeAccountId) {
        youtubeLibraryLoading = true;
    }

    try {
        const response = await ipcRenderer.invoke('youtube.library.list', {
            accountId,
            force,
            maxVideos: options.maxVideos,
            maxPlaylists: options.maxPlaylists,
            maxPlaylistItems: options.maxPlaylistItems
        });

        const library = ensureYouTubeLibraryCache(accountId);
        if (response && response.success) {
            library.fetchedAt = Date.now();
            library.accountId = accountId;
            library.channelId = response.channelId || library.channelId || null;
            library.channelTitle = response.channelTitle || library.channelTitle || null;
            
            // Merge updates instead of replacing - preserve any local updates to video metadata
            const existingVideos = Array.isArray(library.videos) ? library.videos : [];
            const newVideos = Array.isArray(response.videos) ? response.videos : [];
            
            // Create a map of existing videos by videoId to preserve local updates
            const existingVideoMap = new Map();
            existingVideos.forEach(v => {
                if (v && v.videoId) {
                    existingVideoMap.set(v.videoId, v);
                }
            });
            
            // Merge: use new videos from backend, but preserve any locally updated metadata
            library.videos = newVideos.map(newVideo => {
                if (!newVideo || !newVideo.videoId) {
                    return newVideo;
                }
                
                // Ensure videoId is always a string
                const videoId = String(newVideo.videoId).trim();
                if (!videoId) {
                    logger.warn('Skipping video with invalid videoId:', newVideo);
                    return null;
                }
                
                const existing = existingVideoMap.get(videoId);
                if (existing && existing._localUpdate) {
                    // Merge: use backend data but preserve local title/description/privacy if they were updated
                    // CRITICAL: Always preserve the videoId from the backend (it's the source of truth)
                    // Keep the _localUpdate flag so future reloads also preserve it
                    return {
                        ...newVideo, // This includes videoId, url, thumbnails, etc. from backend
                        videoId: videoId, // Ensure videoId is a string
                        title: existing.title || newVideo.title,
                        description: existing.description !== undefined ? existing.description : newVideo.description,
                        privacyStatus: existing.privacyStatus || newVideo.privacyStatus,
                        _localUpdate: true
                    };
                }
                
                // Ensure videoId is a string even for non-updated videos
                return {
                    ...newVideo,
                    videoId: videoId
                };
            }).filter(v => v !== null); // Remove any null entries
            
            // Also add any videos that exist locally but not in the new response (shouldn't happen, but safety)
            existingVideos.forEach(existingVideo => {
                if (existingVideo && existingVideo.videoId && existingVideo._localUpdate) {
                    const found = library.videos.find(v => v && v.videoId === existingVideo.videoId);
                    if (!found) {
                        library.videos.push(existingVideo);
                    }
                }
            });
            
            library.playlists = Array.isArray(response.playlists) ? response.playlists : [];
            library.playlistItems = response.playlistItems && typeof response.playlistItems === 'object'
                ? response.playlistItems
                : {};
            library.errors = null;
            youtubeLibraryByAccount.set(accountId, library);
            youtubeState.libraryByAccount = youtubeLibraryByAccount;
            state.youtube.libraryByAccount = youtubeLibraryByAccount;
            resetYouTubeLibraryIndex(accountId);

            const accountIndex = youtubeAccounts.findIndex(acc => acc.id === accountId);
            if (accountIndex >= 0) {
                const existingAccount = youtubeAccounts[accountIndex];
                youtubeAccounts[accountIndex] = {
                    ...existingAccount,
                    channelTitle: library.channelTitle || existingAccount.channelTitle || existingAccount.channelCustomUrl,
                    channelId: library.channelId || existingAccount.channelId || null
                };
            }
        } else if (response) {
            const errorMessage = response.error || 'Unable to load YouTube library.';
            library.errors = errorMessage;
        }
        if (accountId === currentYouTubeAccountId) {
            youtubeLibrary = library;
            state.youtube.library = youtubeLibrary;
        }
        return library;
    } catch (error) {
        const library = ensureYouTubeLibraryCache(accountId);
        library.errors = error && error.message ? error.message : 'Unable to load YouTube library.';
        if (accountId === currentYouTubeAccountId) {
            youtubeLibrary = library;
            state.youtube.library = youtubeLibrary;
        }
        return library;
    } finally {
        youtubeLibraryLoadingAccounts.set(accountId, false);
        if (accountId === currentYouTubeAccountId) {
            youtubeLibraryLoading = false;
        }
    }
}

let youtubeUploadContext = {
    video: null,
    mode: 'upload',
    existingVideo: null,
    submitting: false,
    accountId: null
};
const INVALID_FILENAME_CHARS = /[\\/:*?"<>|]/;
const TRANSCODER_BASE_URL = (typeof process !== 'undefined' && process.env && process.env.TRANSCODER_BASE_URL) || 'http://127.0.0.1:4455';
const HLS_MIME_TYPE = 'application/vnd.apple.mpegurl';
const hlsSessionCache = state.hlsSessionCache;
const activeHlsSessions = state.activeHlsSessions;

const saveTemplateModal = document.getElementById('saveTemplateModal');
const saveTemplateInput = document.getElementById('saveTemplateInput');
const saveTemplateError = document.getElementById('saveTemplateError');
const saveTemplateConfirmBtn = document.getElementById('saveTemplateConfirmBtn');
const saveTemplateCancelBtn = document.getElementById('saveTemplateCancelBtn');
const saveTemplateCloseBtn = document.getElementById('saveTemplateCloseBtn');
const renameModal = document.getElementById('renameModal');
const renameInput = document.getElementById('renameInput');
const renameExtensionLabel = document.getElementById('renameExtension');
const renamePreviewLabel = document.getElementById('renamePreview');
const renameErrorLabel = document.getElementById('renameError');
const renameConfirmBtn = document.getElementById('renameConfirmBtn');
const renameCancelBtn = document.getElementById('renameCancelBtn');
const renameCloseBtn = document.getElementById('renameCloseBtn');
const renameCurrentNameLabel = document.getElementById('renameCurrentName');
let pendingRenameVideo = null;
let renameExtension = '';
const youtubeAuthStatusLabel = document.getElementById('youtubeAuthStatus');
const youtubeAuthMessage = document.getElementById('youtubeAuthMessage');
const youtubeConnectBtn = document.getElementById('youtubeConnectBtn');
const youtubeDisconnectBtn = document.getElementById('youtubeDisconnectBtn');
const youtubeUploadModal = document.getElementById('youtubeUploadModal');
const youtubeUploadModalTitle = document.getElementById('youtubeUploadModalTitle');
const youtubeUploadTitleInput = document.getElementById('youtubeUploadTitle');
const youtubeUploadDescriptionInput = document.getElementById('youtubeUploadDescription');
const youtubeUploadPrivacySelect = document.getElementById('youtubeUploadPrivacy');
const youtubeUploadEmbeddableCheckbox = document.getElementById('youtubeUploadEmbeddable');
const youtubeUploadSubmitBtn = document.getElementById('youtubeUploadSubmitBtn');
const youtubeUploadCancelBtn = document.getElementById('youtubeUploadCancelBtn');
const youtubeUploadCloseBtn = document.getElementById('youtubeUploadCloseBtn');
const youtubeUploadErrorLabel = document.getElementById('youtubeUploadError');
const youtubeUploadForm = document.getElementById('youtubeUploadForm');
const youtubeAccountSwitcher = document.getElementById('youtubeAccountSwitcher');
const youtubePlayerModal = document.getElementById('youtubePlayerModal');
const youtubePlayerFrame = document.getElementById('youtubePlayerFrame');
const youtubePlayerTitle = document.getElementById('youtubePlayerTitle');
const youtubePlayerCloseBtn = document.getElementById('youtubePlayerCloseBtn');
const youtubePlayerOpenBtn = document.getElementById('youtubePlayerOpenBtn');
const youtubePlayerEditBtn = document.getElementById('youtubePlayerEditBtn');
const youtubePlayerMessage = document.getElementById('youtubePlayerMessage');

let youtubePlayerCurrentUrl = null;
let youtubePlayerCurrentVideoId = null;
let youtubePlayerHadError = false;
function updateAnalyticsState(snapshot) {
    if (!snapshot || typeof snapshot !== 'object') {
        return;
    }
    const previous = state.analytics && typeof state.analytics === 'object' ? state.analytics : {};
    const mergedSession = {
        ...(previous.sessionRecording || {}),
        ...(snapshot.sessionRecording || {})
    };
    state.analytics = {
        ...previous,
        ...snapshot,
        sessionRecording: mergedSession
    };
}

async function initializeAnalytics() {
    try {
        const snapshot = await analyticsClient.bootstrapAnalytics();
        updateAnalyticsState(snapshot);
        if (!snapshot || snapshot.optedIn) {
            return;
        }

        const sessionRecordingEnabled = snapshot.sessionRecording
            ? Boolean(snapshot.sessionRecording.enabled)
            : false;

        const enforcedState = await analyticsClient.setAnalyticsOptIn(true, {
            rotateUserId: !snapshot.userId,
            sessionRecordingEnabled
        });
        updateAnalyticsState(enforcedState);
    } catch (error) {
        logger.warn('Analytics bootstrap enforcement failed:', error && error.message ? error.message : error);
        updateAnalyticsState(analyticsClient.getAnalyticsState());
    }
}

function needsLiveTranscode(video) {
    if (!video || !video.extension) {
        return false;
    }
    const ext = String(video.extension).toLowerCase();
    return ext === 'mov';
}

function setPlayerStatus(statusElement, message, type = 'info') {
    if (!statusElement) {
        return;
    }
    if (!message) {
        statusElement.style.display = 'none';
        statusElement.textContent = '';
        return;
    }
    statusElement.textContent = message;
    statusElement.style.display = 'block';
    statusElement.style.backgroundColor = type === 'error'
        ? 'rgba(128, 0, 0, 0.8)'
        : 'rgba(0, 0, 0, 0.65)';
}

function showPlaybackError(errorElement, message) {
    if (!errorElement) {
        return;
    }
    if (!message) {
        errorElement.style.display = 'none';
        errorElement.textContent = '';
        return;
    }
    errorElement.textContent = message;
    errorElement.style.display = 'block';
}

async function ensureHlsSession(sourcePath, options = {}) {
    const cacheKey = sourcePath;
    const forceNew = Boolean(options.forceNew);

    if (!forceNew && hlsSessionCache.has(cacheKey)) {
        return hlsSessionCache.get(cacheKey);
    }

    try {
        const response = await fetch(`${TRANSCODER_BASE_URL}/hls/session`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                sourcePath,
                forceNew
            })
        });

        if (!response.ok) {
            throw new Error(`Transcoder responded with status ${response.status}`);
        }

        const payload = await response.json();
        payload.sourcePath = sourcePath;
        hlsSessionCache.set(cacheKey, payload);
        return payload;
    } catch (error) {
        const message = error && error.message ? error.message : 'Failed to reach the live transcoder service.';
        throw new Error(message);
    }
}

async function setupVideoPlayback(videoElement, video, { statusElement, errorElement } = {}) {
    if (!videoElement || !video) {
        return;
    }

    showPlaybackError(errorElement, '');
    setPlayerStatus(statusElement, '', 'info');

    videoElement.pause();
    videoElement.removeAttribute('src');
    videoElement.load();

    const isTranscode = needsLiveTranscode(video);

    if (!isTranscode) {
        const normalizedPath = normalizeFilePath(video.path);
        videoElement.src = `file://${normalizedPath}`;
        videoElement.load();
        setPlayerStatus(statusElement, '', 'info');
        return;
    }

    setPlayerStatus(statusElement, 'Preparing HEVC stream…', 'info');

    try {
        const session = await ensureHlsSession(video.path);
        activeHlsSessions.set(videoElement, session);

        const manifestUrl = `${TRANSCODER_BASE_URL}${session.manifestUrl}`;
        if (videoElement.canPlayType(HLS_MIME_TYPE)) {
            videoElement.src = manifestUrl;
            videoElement.load();
        } else if (window.Hls && typeof window.Hls.isSupported === 'function' && window.Hls.isSupported()) {
            const hls = new window.Hls();
            hls.loadSource(manifestUrl);
            hls.attachMedia(videoElement);
            videoElement._hls = hls;
        } else {
            throw new Error('HLS playback is not supported in this environment.');
        }

        videoElement.addEventListener('loadedmetadata', () => {
            setPlayerStatus(statusElement, '', 'info');
        }, { once: true });
    } catch (error) {
        activeHlsSessions.delete(videoElement);
        setPlayerStatus(statusElement, '', 'info');
        showPlaybackError(errorElement, 'Unable to play HEVC video. Start the live transcoder service and try again.');
        trackEvent('viewer_transcode_failed', {
            ...getVideoAnalyticsProps(video),
            error_message: error && error.message ? error.message : 'Unknown error'
        });
        throw error;
    }
}

function cleanupVideoElement(videoElement) {
    if (!videoElement) {
        return;
    }

    const session = activeHlsSessions.get(videoElement);
    if (session) {
        activeHlsSessions.delete(videoElement);
        hlsSessionCache.delete(session.sourcePath);
        fetch(`${TRANSCODER_BASE_URL}/hls/session/${session.sessionId}`, {
            method: 'DELETE'
        }).catch(() => {});
    }

    if (videoElement._hls && typeof videoElement._hls.destroy === 'function') {
        try {
            videoElement._hls.destroy();
        } catch (error) {
            // ignore hls teardown errors
        }
        delete videoElement._hls;
    }

    videoElement.pause();
    videoElement.removeAttribute('src');
    videoElement.load();
}

function cleanupPlaybackContainer() {
    const playerContainer = document.getElementById('playerContainer');
    if (!playerContainer) {
        return;
    }
    const existingVideoElement = playerContainer.querySelector('video');
    if (existingVideoElement) {
        cleanupVideoElement(existingVideoElement);
    }
}

// Load settings on startup
async function loadSettings() {
    settings = await ipcRenderer.invoke('get-settings');
    state.settings = settings;
    renderDirectoryList();
}

// Render directory list in settings
function renderDirectoryList() {
    const list = document.getElementById('directoryList');
    if (settings.monitoredDirectories.length === 0) {
        list.innerHTML = '<p style="color: #A7A9AB; text-align: center; padding: 20px;">No directories added</p>';
        return;
    }
    list.innerHTML = settings.monitoredDirectories.map((dir, index) => {
        const dirPath = typeof dir === 'string' ? dir : dir.path;
        const dirName = typeof dir === 'string' ? dirPath.split(/[\\/]/).pop() : dir.name;
        return `
        <div class="directory-item">
            <div class="directory-item-row">
                <input type="text" class="directory-name-input" 
                       value="${dirName}" 
                       placeholder="Directory name"
                       data-index="${index}">
                <button class="directory-remove" data-index="${index}">Remove</button>
            </div>
            <div class="directory-path">${dirPath}</div>
        </div>
    `;
    }).join('');

    // Add event listeners for name inputs
    list.querySelectorAll('.directory-name-input').forEach(input => {
        input.addEventListener('change', (e) => {
            const index = parseInt(e.target.dataset.index);
            const dirObj = settings.monitoredDirectories[index];
            if (typeof dirObj === 'string') {
                const dirPath = dirObj;
                settings.monitoredDirectories[index] = {
                    path: dirPath,
                    name: e.target.value || dirPath.split(/[\\/]/).pop()
                };
            } else {
                dirObj.name = e.target.value || dirObj.path.split(/[\\/]/).pop();
            }
        });
    });

    list.querySelectorAll('.directory-remove').forEach(btn => {
        btn.addEventListener('click', () => {
            const index = parseInt(btn.dataset.index);
            const targetDir = settings.monitoredDirectories[index];
            const dirPath = typeof targetDir === 'string' ? targetDir : (targetDir ? targetDir.path : '');
            const dirName = typeof targetDir === 'string'
                ? dirPath.split(/[\\/]/).pop()
                : (targetDir && targetDir.name) || (dirPath ? dirPath.split(/[\\/]/).pop() : '');
            settings.monitoredDirectories.splice(index, 1);
            const metrics = summarizeDirectoryPath(dirPath);
            trackEvent('settings_directory_removed', {
                ...metrics,
                directory_label_length: dirName ? Math.min(dirName.length, 64) : 0,
                total_directories: settings.monitoredDirectories.length
            });
            renderDirectoryList();
        });
    });
}

function renderYouTubeAuthSection() {
    if (!youtubeAuthStatusLabel || !youtubeConnectBtn || !youtubeDisconnectBtn || !youtubeAuthMessage) {
        return;
    }

    const { status, inProgress, manualAuthUrl, lastError } = youtubeAuthState;
    const isConnected = Boolean(status && status.authenticated);

    youtubeAuthStatusLabel.textContent = isConnected ? 'Connected to YouTube' : 'Not connected to YouTube';
    youtubeAuthStatusLabel.classList.toggle('youtube-auth-status-connected', isConnected);
    youtubeAuthStatusLabel.classList.toggle('youtube-auth-status-disconnected', !isConnected);

    youtubeConnectBtn.disabled = inProgress;
    youtubeDisconnectBtn.disabled = inProgress || !isConnected;
    youtubeConnectBtn.textContent = isConnected ? 'Add YouTube Account' : 'Connect YouTube Account';

    youtubeAuthMessage.innerHTML = '';

    if (inProgress) {
        const message = document.createElement('span');
        if (manualAuthUrl) {
            message.textContent = 'Waiting for authorization. If your browser did not open automatically, use the link below.';
        } else {
            message.textContent = 'Waiting for authorization in your browser…';
        }
        youtubeAuthMessage.appendChild(message);
    } else if (lastError) {
        const message = document.createElement('span');
        message.className = 'youtube-auth-error';
        message.textContent = lastError;
        youtubeAuthMessage.appendChild(message);
    } else if (isConnected) {
        const message = document.createElement('span');
        const accountNames = Array.isArray(youtubeAccounts) && youtubeAccounts.length > 0
            ? youtubeAccounts.map(acc => getYouTubeAccountLabel(acc)).join(', ')
            : 'YouTube';
        if (status && status.expiresAt) {
            const expiry = new Date(status.expiresAt);
            message.textContent = `Connected to ${accountNames}. Access token valid until ${expiry.toLocaleString()}.`;
        } else {
            message.textContent = `Connected to ${accountNames}. Ready to upload and manage videos.`;
        }
        youtubeAuthMessage.appendChild(message);

        if (Array.isArray(youtubeAccounts) && youtubeAccounts.length > 0) {
            const list = document.createElement('div');
            list.className = 'youtube-account-list';
            youtubeAccounts.forEach((account) => {
                if (!account || !account.id) {
                    return;
                }
                const item = document.createElement('div');
                item.className = 'youtube-account-list-item';
                const label = document.createElement('span');
                label.textContent = getYouTubeAccountLabel(account);
                item.appendChild(label);

                const actions = document.createElement('div');
                actions.className = 'youtube-account-actions';

                if (youtubeAccounts.length > 1) {
                    const removeBtn = document.createElement('button');
                    removeBtn.className = 'btn btn-link';
                    removeBtn.type = 'button';
                    removeBtn.textContent = 'Remove';
                    removeBtn.addEventListener('click', () => {
                        removeYouTubeAccount(account.id);
                    });
                    actions.appendChild(removeBtn);
                }

            if (youtubeActiveAccountId === account.id) {
                    const badge = document.createElement('span');
                    badge.className = 'youtube-account-badge';
                    badge.textContent = 'Active';
                    actions.appendChild(badge);
                } else {
                    const activateBtn = document.createElement('button');
                    activateBtn.className = 'btn btn-link';
                    activateBtn.type = 'button';
                    activateBtn.textContent = 'Set Active';
                    activateBtn.addEventListener('click', () => {
                        selectYouTubeAccount(account.id);
                    });
                    actions.appendChild(activateBtn);
                }

                item.appendChild(actions);
                list.appendChild(item);
            });
            youtubeAuthMessage.appendChild(list);
            
            // Add cache reset button
            const cacheResetContainer = document.createElement('div');
            cacheResetContainer.className = 'youtube-cache-actions';
            cacheResetContainer.style.marginTop = '15px';
            cacheResetContainer.style.paddingTop = '15px';
            cacheResetContainer.style.borderTop = '1px solid rgba(255, 255, 255, 0.1)';
            
            const cacheResetBtn = document.createElement('button');
            cacheResetBtn.className = 'btn btn-secondary';
            cacheResetBtn.type = 'button';
            cacheResetBtn.textContent = 'Reset Library Cache';
            cacheResetBtn.title = 'Clear all cached YouTube library data. The library will be reloaded on next access.';
            cacheResetBtn.addEventListener('click', () => {
                const confirmed = confirm('Reset YouTube library cache? This will clear all cached video data. The library will be reloaded from YouTube on next access.');
                if (confirmed) {
                    resetYouTubeLibraryCache();
                    // Reload the library for the current account if authenticated
                    if (currentYouTubeAccountId) {
                        loadYouTubeLibraryForAccount(currentYouTubeAccountId, { force: true, cacheMs: 0 }).catch((error) => {
                            logger.warn('Failed to reload YouTube library after cache reset:', error && error.message ? error.message : error);
                        });
                    }
                    renderYouTubeAuthSection();
                }
            });
            cacheResetContainer.appendChild(cacheResetBtn);
            youtubeAuthMessage.appendChild(cacheResetContainer);
        }
    }

    if (manualAuthUrl) {
        const linkContainer = document.createElement('div');
        linkContainer.className = 'youtube-auth-link';
        const link = document.createElement('a');
        link.href = manualAuthUrl;
        link.textContent = 'Open YouTube authorization link';
        link.target = '_blank';
        link.rel = 'noopener noreferrer';
        linkContainer.appendChild(link);
        youtubeAuthMessage.appendChild(linkContainer);
    }
}

async function refreshYouTubeAuthStatus(options = {}) {
    if (!ipcRenderer) {
        return;
    }

    const { preserveInProgress = false, hydrateIfAuthenticated = false } = options;
    // Check if we had a real auth state before (not just the initial default state)
    // Initial state has status: { authenticated: false }, so we check if we had actual account data
    const wasAuthenticated = Boolean(
        youtubeAuthState && 
        youtubeAuthState.status && 
        youtubeAuthState.status.authenticated === true
    );
    // Check if we had a real auth check before (has accounts array or activeAccountId, not just default state)
    const hadRealAuthCheck = Boolean(
        youtubeAuthState && 
        youtubeAuthState.status && 
        (
            (Array.isArray(youtubeAuthState.status.accounts) && youtubeAuthState.status.accounts.length > 0) ||
            youtubeAuthState.status.activeAccountId
        )
    );

    try {
        const result = await ipcRenderer.invoke('youtube.auth.status');
        if (result && result.success && result.status) {
            youtubeAuthState.status = result.status;
            if (!preserveInProgress) {
                youtubeAuthState.lastError = null;
                youtubeAuthState.manualAuthUrl = null;
            }
        } else if (!preserveInProgress && result && !result.success) {
            youtubeAuthState.lastError = result.error || 'Unable to determine YouTube authentication status.';
        }
    } catch (error) {
        if (!preserveInProgress) {
            youtubeAuthState.lastError = error && error.message ? error.message : 'Failed to reach YouTube authentication service.';
        }
    }

    if (!preserveInProgress) {
        youtubeAuthState.inProgress = false;
    }

    const isAuthenticated = Boolean(youtubeAuthState.status && youtubeAuthState.status.authenticated);
    // Only treat as "just authenticated" if we had a real previous auth check and it was NOT authenticated
    // This prevents treating "initial load with authenticated user" as "just became authenticated"
    if (isAuthenticated && hadRealAuthCheck && !wasAuthenticated) {
        // User just became authenticated during this session - hydrate with force
        logger.log('[Renderer] refreshYouTubeAuthStatus: User just became authenticated, calling hydrateYouTubeState with force');
        await hydrateYouTubeState({ force: true });
        trackEvent('youtube_auth_authenticated', {
            account_count: Array.isArray(youtubeAccounts) ? youtubeAccounts.length : 0
        });
    } else if (isAuthenticated && hydrateIfAuthenticated) {
        // User was already authenticated (or just authenticated on initial load) - hydrate without force
        logger.log('[Renderer] refreshYouTubeAuthStatus: User authenticated, calling hydrateYouTubeState without force');
        await hydrateYouTubeState();
    } else if (!isAuthenticated && wasAuthenticated) {
        youtubeLibrary = {
            fetchedAt: 0,
            videos: [],
            playlists: [],
            playlistItems: {},
            errors: null
        };
        youtubeState.library = youtubeLibrary;
        state.youtube.library = youtubeLibrary;
        resetYouTubeLibraryIndex();
        if (isYouTubeTreeKey(currentTreeSelection)) {
            filterAndRenderGallery(currentTreeSelection);
        }
        trackEvent('youtube_auth_signed_out', {
            reason: 'token_expired'
        });
    }

    renderYouTubeAuthSection();
}

async function startYouTubeAuthFlow() {
    if (youtubeAuthState.inProgress) {
        return;
    }

    youtubeAuthState.inProgress = true;
    youtubeAuthState.lastError = null;
    youtubeAuthState.manualAuthUrl = null;
    renderYouTubeAuthSection();
    trackEvent('youtube_auth_flow_started', {
        existing_accounts: Array.isArray(youtubeAccounts) ? youtubeAccounts.length : 0
    });

    try {
        const result = await ipcRenderer.invoke('youtube.auth.start');
        if (!result || !result.success) {
            throw new Error(result && result.error ? result.error : 'Failed to start YouTube authentication.');
        }

        if (result.browserOpened) {
            youtubeAuthState.manualAuthUrl = null;
        } else if (result.authUrl) {
            youtubeAuthState.manualAuthUrl = result.authUrl;
        }
        trackEvent('youtube_auth_flow_triggered', {
            manual_step_required: result.browserOpened ? 0 : (result.authUrl ? 1 : 0)
        });
    } catch (error) {
        youtubeAuthState.inProgress = false;
        youtubeAuthState.lastError = error && error.message ? error.message : 'Failed to initiate YouTube authentication.';
        trackEvent('youtube_auth_flow_failed', {
            error_message: youtubeAuthState.lastError
        });
    }

    renderYouTubeAuthSection();
}

async function disconnectYouTubeAccount() {
    if (youtubeAuthState.inProgress) {
        return;
    }

    youtubeAuthState.inProgress = true;
    youtubeAuthState.lastError = null;
    youtubeAuthState.manualAuthUrl = null;
    renderYouTubeAuthSection();

    try {
        const result = await ipcRenderer.invoke('youtube.auth.logout');
        if (!result || !result.success) {
            throw new Error(result && result.error ? result.error : 'Failed to disconnect YouTube account.');
        }

        if (result.status) {
            youtubeAuthState.status = result.status;
        } else {
            youtubeAuthState.status = { authenticated: false };
        }
        trackEvent('youtube_auth_disconnected', {
            success: 1,
            remaining_accounts: Array.isArray(result.status && result.status.accounts) ? result.status.accounts.length : 0
        });
    } catch (error) {
        youtubeAuthState.lastError = error && error.message ? error.message : 'Failed to disconnect YouTube account.';
        trackEvent('youtube_auth_disconnected', {
            success: 0,
            error_message: youtubeAuthState.lastError
        });
    } finally {
        youtubeAuthState.inProgress = false;
        renderYouTubeAuthSection();
    }
}

async function selectYouTubeAccount(accountId) {
    if (!accountId || !ipcRenderer) {
        return;
    }
    try {
        const response = await ipcRenderer.invoke('youtube.accounts.setActive', { accountId });
        if (response && response.success) {
            youtubeAuthState.status = response.status || youtubeAuthState.status;
            // Don't force refresh when switching accounts - check cache first
            await hydrateYouTubeState();
            renderYouTubeAuthSection();
            trackEvent('youtube_account_selected', {
                account_id_present: accountId ? 1 : 0
            });
        } else if (response && response.error) {
            alert(response.error);
            trackEvent('youtube_account_select_failed', {
                error_message: response.error || 'unknown'
            });
        }
    } catch (error) {
        alert(error && error.message ? error.message : 'Failed to set active YouTube account.');
        trackEvent('youtube_account_select_failed', {
            error_message: error && error.message ? error.message : 'unknown'
        });
    }
}

async function removeYouTubeAccount(accountId) {
    if (!accountId || !ipcRenderer) {
        return;
    }
    const confirmed = confirm('Remove this YouTube account? You will need to re-authenticate to use it again.');
    if (!confirmed) {
        return;
    }

    try {
        const response = await ipcRenderer.invoke('youtube.accounts.remove', { accountId });
        if (response && response.success) {
            youtubeAuthState.status = response.status || youtubeAuthState.status;
            await hydrateYouTubeState({ force: true });
            renderYouTubeAuthSection();
            trackEvent('youtube_account_removed', {
                account_id_present: accountId ? 1 : 0
            });
        } else if (response && response.error) {
            alert(response.error);
            trackEvent('youtube_account_remove_failed', {
                error_message: response.error || 'unknown'
            });
        }
    } catch (error) {
        alert(error && error.message ? error.message : 'Failed to remove YouTube account.');
        trackEvent('youtube_account_remove_failed', {
            error_message: error && error.message ? error.message : 'unknown'
        });
    }
}

// Settings modal - opened from File menu
ipcRenderer.on('open-settings', () => {
    document.getElementById('settingsModal').classList.add('active');
    loadSettings();
    refreshYouTubeAuthStatus({ preserveInProgress: youtubeAuthState.inProgress });
});

document.getElementById('closeSettingsBtn').addEventListener('click', () => {
    document.getElementById('settingsModal').classList.remove('active');
});

document.getElementById('cancelSettingsBtn').addEventListener('click', () => {
    document.getElementById('settingsModal').classList.remove('active');
});

document.getElementById('addDirectoryBtn').addEventListener('click', async () => {
    const dir = await ipcRenderer.invoke('select-directory');
    if (dir) {
        // Check if directory already exists
        const exists = settings.monitoredDirectories.some(d => {
            const dPath = typeof d === 'string' ? d : d.path;
            return dPath === dir;
        });
        if (!exists) {
            const dirName = dir.split(/[\\/]/).pop();
            settings.monitoredDirectories.push({
                path: dir,
                name: dirName
            });
            const metrics = summarizeDirectoryPath(dir);
            trackEvent('settings_directory_added', {
                ...metrics,
                directory_label_length: Math.min(dirName.length, 64),
                total_directories: settings.monitoredDirectories.length
            });
            renderDirectoryList();
        } else {
            const metrics = summarizeDirectoryPath(dir);
            trackEvent('settings_directory_add_skipped', {
                ...metrics,
                directory_label_length: Math.min(dir.split(/[\\/]/).pop().length, 64),
                reason: 'duplicate'
            });
        }
    }
});

document.getElementById('saveSettingsBtn').addEventListener('click', async () => {
    await ipcRenderer.invoke('save-settings', settings);
    trackEvent('settings_saved', {
        directory_count: settings.monitoredDirectories.length,
        youtube_connected: youtubeAuthState.status && youtubeAuthState.status.authenticated ? 1 : 0,
        analytics_opted_in: state.analytics && state.analytics.optedIn ? 1 : 0
    });
    document.getElementById('settingsModal').classList.remove('active');
    // Reload videos
    await scanAllDirectories();
});

if (youtubeConnectBtn) {
    youtubeConnectBtn.addEventListener('click', () => {
        startYouTubeAuthFlow();
    });
}

if (youtubeDisconnectBtn) {
    youtubeDisconnectBtn.addEventListener('click', () => {
        disconnectYouTubeAccount();
    });
}

function ensureYoutubeAuthBeforeAction() {
    if (youtubeAuthState.status && youtubeAuthState.status.authenticated) {
        return true;
    }
    const proceed = confirm('You need to connect your YouTube account before using YouTube features. Open settings now?');
    if (proceed) {
        const settingsModal = document.getElementById('settingsModal');
        if (settingsModal) {
            settingsModal.classList.add('active');
        }
        loadSettings();
        refreshYouTubeAuthStatus({ preserveInProgress: youtubeAuthState.inProgress });
    }
    return false;
}

function ensureYouTubePrivacyOptions() {
    if (!youtubeUploadPrivacySelect) {
        return;
    }
    if (youtubeUploadPrivacySelect.options.length === 0) {
        youtubeUploadPrivacySelect.innerHTML = '';
        YOUTUBE_PRIVACY_OPTIONS.forEach((value) => {
            const option = document.createElement('option');
            option.value = value;
            option.textContent = value.charAt(0).toUpperCase() + value.slice(1);
            youtubeUploadPrivacySelect.appendChild(option);
        });
    }
}

function ensureYouTubeAccountOptions(selectedAccountId = null) {
    const resolvedAccountId = getResolvedYouTubeAccountId(
        selectedAccountId
        || (youtubeUploadContext && youtubeUploadContext.accountId)
    );
    syncUploadContextAccount(resolvedAccountId);
    return resolvedAccountId;
}

function setYouTubeUploadError(message = '') {
    if (!youtubeUploadErrorLabel) {
        return;
    }
    youtubeUploadErrorLabel.textContent = message;
    youtubeUploadErrorLabel.style.display = message ? 'block' : 'none';
}

function resetYouTubeUploadModal() {
    youtubeUploadContext = {
        video: null,
        mode: 'upload',
        existingVideo: null,
        submitting: false,
        accountId: null,
        preservedPlayerState: null
    };

    if (youtubeUploadForm) {
        youtubeUploadForm.reset();
    } else {
        if (youtubeUploadTitleInput) {
            youtubeUploadTitleInput.value = '';
        }
        if (youtubeUploadDescriptionInput) {
            youtubeUploadDescriptionInput.value = '';
        }
        if (youtubeUploadPrivacySelect) {
            youtubeUploadPrivacySelect.value = 'unlisted';
        }
    }

    ensureYouTubeAccountOptions();

    if (youtubeUploadSubmitBtn) {
        youtubeUploadSubmitBtn.disabled = false;
        youtubeUploadSubmitBtn.textContent = 'Start Upload';
    }

    setYouTubeUploadError('');
}

function closeYouTubeUploadModal() {
    if (youtubeUploadModal) {
        youtubeUploadModal.classList.remove('active');
    }
    resetYouTubeUploadModal();
}

function getYouTubeRecord(videoPath, accountId = getResolvedYouTubeAccountId()) {
    const record = youtubeState.videos.get(videoPath) || null;
    if (!record) {
        return null;
    }
    // Records now always have accountId (filtered by active account when loaded)
    // So we just check if accountId matches
    if (!accountId) {
        return record;
    }
    return record.accountId === accountId ? record : null;
}

function updateYouTubeRecord(videoPath, updates = {}, accountId = getResolvedYouTubeAccountId()) {
    const previous = youtubeState.videos.get(videoPath) || {};
    const resolvedAccountId = updates.accountId != null
        ? updates.accountId
        : (previous.accountId != null ? previous.accountId : accountId);
    
    // Only update if accountId matches (preserve other accounts' data)
    // Since records are now filtered by active account, this should always match
    if (previous.accountId && previous.accountId !== resolvedAccountId) {
        // Account mismatch - don't overwrite other account's data
        // This shouldn't happen since records are filtered, but be safe
        return previous;
    }
    
    const merged = {
        ...previous,
        ...updates,
        accountId: resolvedAccountId,
        updatedAt: Date.now()
    };
    youtubeState.videos.set(videoPath, merged);
    return merged;
}

function removeYouTubeRecord(videoPath) {
    youtubeState.videos.delete(videoPath);
}

function upsertYouTubeJob(jobUpdate = {}) {
    if (!jobUpdate || !jobUpdate.id) {
        return null;
    }
    const existing = youtubeState.jobs.get(jobUpdate.id) || {};
    const merged = {
        ...existing,
        ...jobUpdate
    };
    youtubeState.jobs.set(jobUpdate.id, merged);
    if (merged.videoPath) {
        const jobKey = buildYouTubeJobKey(merged.videoPath, merged.accountId);
        youtubeState.jobsByVideoPath.set(jobKey, merged.id);
    }
    return merged;
}

function removeYouTubeJob(jobId) {
    if (!jobId) {
        return;
    }
    const job = youtubeState.jobs.get(jobId);
    if (job && job.videoPath) {
        const jobKey = buildYouTubeJobKey(job.videoPath, job.accountId);
        youtubeState.jobsByVideoPath.delete(jobKey);
    }
    youtubeState.jobs.delete(jobId);
}

function getYouTubeJobByVideoPath(videoPath, accountId = currentYouTubeAccountId || youtubeActiveAccountId || null) {
    const jobKey = buildYouTubeJobKey(videoPath, accountId);
    const jobId = youtubeState.jobsByVideoPath.get(jobKey);
    if (!jobId) {
        return null;
    }
    return youtubeState.jobs.get(jobId) || null;
}

function getFileNameFromPath(filePath) {
    if (!filePath) {
        return '';
    }
    const segments = String(filePath).split(/[\\/]+/);
    return segments.length > 0 ? segments[segments.length - 1] : String(filePath);
}

function getYouTubeRecordTimestamp(record = {}) {
    if (!record || typeof record !== 'object') {
        return 0;
    }

    const { updatedAt, publishedAt } = record;
    if (typeof updatedAt === 'number' && Number.isFinite(updatedAt)) {
        return updatedAt;
    }

    if (updatedAt) {
        const parsedUpdated = Date.parse(updatedAt);
        if (!Number.isNaN(parsedUpdated)) {
            return parsedUpdated;
        }
    }

    if (publishedAt) {
        const parsedPublished = Date.parse(publishedAt);
        if (!Number.isNaN(parsedPublished)) {
            return parsedPublished;
        }
    }

    return 0;
}

function formatYouTubeDate(value) {
    if (!value) {
        return null;
    }
    const date = typeof value === 'number' && Number.isFinite(value)
        ? new Date(value)
        : new Date(String(value));
    if (Number.isNaN(date.getTime())) {
        return null;
    }
    return date.toLocaleDateString(undefined, { month: 'short', day: 'numeric', year: 'numeric' });
}

function escapeHtml(value) {
    if (value === null || value === undefined) {
        return '';
    }
    return String(value)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}

function getYouTubeStatusMeta(record = {}, job = null) {
    let status = null;
    let progress = null;

    if (job && job.status) {
        status = String(job.status).toLowerCase();
        if (typeof job.progress === 'number' && Number.isFinite(job.progress)) {
            progress = Math.max(0, Math.min(100, job.progress));
        }
    } else if (record && record.status) {
        status = String(record.status).toLowerCase();
    } else if (record && record.videoId) {
        status = 'uploaded';
    }

    switch (status) {
        case 'uploading':
            return {
                className: 'uploading',
                label: progress !== null ? `Uploading ${Math.round(progress)}%` : 'Uploading'
            };
        case 'processing':
            return {
                className: 'processing',
                label: 'Processing'
            };
        case 'completed':
            return {
                className: 'uploaded',
                label: 'Completed'
            };
        case 'uploaded':
            return {
                className: 'uploaded',
                label: 'Uploaded'
            };
        case 'failed':
        case 'error':
            return {
                className: 'failed',
                label: 'Failed'
            };
        case 'removing':
            return {
                className: 'removing',
                label: 'Removing'
            };
        case 'pending':
            return {
                className: 'pending',
                label: 'Pending'
            };
        default:
            return {
                className: 'unknown',
                label: 'Not Uploaded'
            };
    }
}

function resetYouTubeLibraryIndex(accountId = currentYouTubeAccountId || (youtubeLibrary && youtubeLibrary.accountId) || null) {
    const key = accountId || '__default__';
    youtubeLibraryIndexByAccount.delete(key);
    if ((youtubeLibrary && (youtubeLibrary.accountId || '__default__')) === key) {
        youtubeLibraryVideoIndex = null;
    }
    if (!accountId) {
        youtubeLibraryVideoIndex = null;
    }
}

function resetYouTubeLibraryCache() {
    // Clear all library caches
    youtubeLibraryByAccount.clear();
    youtubeLibraryIndexByAccount.clear();
    
    // Reset current library
    youtubeLibrary = createEmptyYouTubeLibrary();
    if (currentYouTubeAccountId) {
        youtubeLibrary.accountId = currentYouTubeAccountId;
        youtubeLibraryByAccount.set(currentYouTubeAccountId, youtubeLibrary);
    }
    youtubeLibraryVideoIndex = null;
    
    // Update state
    youtubeState.libraryByAccount = youtubeLibraryByAccount;
    youtubeState.libraryIndexByAccount = youtubeLibraryIndexByAccount;
    youtubeState.library = youtubeLibrary;
    state.youtube.libraryByAccount = youtubeLibraryByAccount;
    state.youtube.libraryIndexByAccount = youtubeLibraryIndexByAccount;
    state.youtube.library = youtubeLibrary;
    
    // Refresh the gallery if we're viewing YouTube content
    if (isYouTubeTreeKey(currentTreeSelection)) {
        filterAndRenderGallery(currentTreeSelection);
    }
    
    trackEvent('youtube_cache_reset', {
        account_id: currentYouTubeAccountId || null
    });
}

function ensureYouTubeLibraryIndex(targetLibrary = youtubeLibrary) {
    if (!targetLibrary) {
        return new Map();
    }
    const accountKey = targetLibrary.accountId || currentYouTubeAccountId || '__default__';
    const existingIndex = youtubeLibraryIndexByAccount.get(accountKey);
    if (existingIndex && existingIndex instanceof Map) {
        youtubeLibraryVideoIndex = existingIndex;
        return existingIndex;
    }

    const index = new Map();
    const libraryVideos = Array.isArray(targetLibrary.videos) ? targetLibrary.videos : [];
    for (const video of libraryVideos) {
        if (video && video.videoId) {
            index.set(video.videoId, video);
        }
    }
    youtubeLibraryIndexByAccount.set(accountKey, index);
    if ((youtubeLibrary && (youtubeLibrary.accountId || currentYouTubeAccountId || '__default__')) === accountKey) {
        youtubeLibraryVideoIndex = index;
    }
    return index;
}

function getYouTubeLibraryVideos(targetLibrary = youtubeLibrary) {
    if (!targetLibrary || !Array.isArray(targetLibrary.videos)) {
        return [];
    }
    return targetLibrary.videos;
}

function getYouTubeLibraryPlaylists(targetLibrary = youtubeLibrary) {
    if (!targetLibrary || !Array.isArray(targetLibrary.playlists)) {
        return [];
    }
    return targetLibrary.playlists;
}

function getYouTubePlaylistsByType(type, targetLibrary = youtubeLibrary) {
    return getYouTubeLibraryPlaylists(targetLibrary).filter(playlist => {
        if (!playlist || !playlist.type) {
            return type === 'playlists';
        }
        if (type === 'podcasts') {
            return playlist.type === 'podcast';
        }
        return playlist.type !== 'podcast';
    });
}

function getYouTubePlaylistById(playlistId, targetLibrary = youtubeLibrary) {
    return getYouTubeLibraryPlaylists(targetLibrary).find(playlist => playlist && playlist.id === playlistId) || null;
}

function getYouTubePlaylistItems(playlistId, targetLibrary = youtubeLibrary) {
    if (!playlistId || !targetLibrary || typeof targetLibrary.playlistItems !== 'object') {
        return [];
    }
    const items = targetLibrary.playlistItems[playlistId];
    if (!Array.isArray(items)) {
        return [];
    }
    return items.filter(item => item && item.videoId);
}

function getYouTubeVideosForPlaylist(playlistId, targetLibrary = youtubeLibrary) {
    const index = ensureYouTubeLibraryIndex(targetLibrary);
    const items = getYouTubePlaylistItems(playlistId, targetLibrary);
    const videosForPlaylist = [];
    for (const item of items) {
        const video = index.get(item.videoId);
        if (video) {
            videosForPlaylist.push({
                ...video,
                position: item.position,
                playlistId
            });
        }
    }
    videosForPlaylist.sort((a, b) => {
        const posA = Number.isFinite(a.position) ? a.position : Number.MAX_SAFE_INTEGER;
        const posB = Number.isFinite(b.position) ? b.position : Number.MAX_SAFE_INTEGER;
        return posA - posB;
    });
    return videosForPlaylist;
}

function getYouTubeVideosByCategory(category, targetLibrary = youtubeLibrary) {
    const normalized = (category || '').toLowerCase();
    const videosList = getYouTubeLibraryVideos(targetLibrary);
    if (!normalized) {
        return videosList.slice();
    }
    return videosList.filter(video => {
        const candidate = (video && video.type) ? String(video.type).toLowerCase() : 'videos';
        if (normalized === 'videos') {
            return candidate === 'videos' || candidate === 'video';
        }
        if (normalized === 'shorts') {
            return candidate === 'shorts' || candidate === 'short';
        }
        if (normalized === 'live') {
            return candidate === 'live';
        }
        return candidate === normalized;
    });
}

function applySearchFilterToYouTubeItems(items) {
    if (!searchQuery) {
        return items;
    }
    const query = searchQuery.toLowerCase();
    return items.filter(item => {
        if (!item) {
            return false;
        }
        const title = (item.title || '').toLowerCase();
        const description = (item.description || '').toLowerCase();
        const videoId = (item.videoId || '').toLowerCase();
        const extra = (item.channelTitle || '').toLowerCase();
        return title.includes(query) || description.includes(query) || videoId.includes(query) || extra.includes(query);
    });
}


function getVideoBaseName(name) {
    if (!name) {
        return '';
    }
    const lastDot = name.lastIndexOf('.');
    if (lastDot <= 0) {
        return name;
    }
    return name.slice(0, lastDot);
}

function populateYouTubeUploadForm(video, mode, record) {
    if (youtubeUploadModalTitle) {
        youtubeUploadModalTitle.textContent = mode === 'update' ? 'Update YouTube Metadata' : 'Upload to YouTube';
    }

    if (youtubeUploadTitleInput) {
        const defaultTitle = record && record.title ? record.title : getVideoBaseName(video.name);
        youtubeUploadTitleInput.value = defaultTitle || getVideoBaseName(video.name);
    }

    if (youtubeUploadDescriptionInput) {
        youtubeUploadDescriptionInput.value = record && record.description ? record.description : '';
    }

    if (youtubeUploadPrivacySelect) {
        const desiredPrivacy = record && record.privacy ? record.privacy : 'unlisted';
        youtubeUploadPrivacySelect.value = YOUTUBE_PRIVACY_OPTIONS.includes(desiredPrivacy) ? desiredPrivacy : 'unlisted';
    }

    if (youtubeUploadEmbeddableCheckbox) {
        // Default to true if not specified (YouTube's default)
        const embeddable = record && record.embeddable !== undefined ? record.embeddable : true;
        youtubeUploadEmbeddableCheckbox.checked = embeddable;
    }

    ensureYouTubeAccountOptions(record && record.accountId ? record.accountId : null);

    if (youtubeUploadSubmitBtn) {
        youtubeUploadSubmitBtn.textContent = mode === 'update' ? 'Save Changes' : 'Start Upload';
        youtubeUploadSubmitBtn.disabled = false;
    }
}

function openYouTubeUploadModal(video, mode = 'upload') {
    if (!video) {
        return;
    }
    if (!ensureYoutubeAuthBeforeAction()) {
        return;
    }
    
    // Determine if this is a YouTube library video (no local file) or local video with YouTube record
    const isYouTubeLibraryVideo = video.isYouTubeLibraryVideo || (video.path && video.path.startsWith('__youtube__'));
    const videoId = video.youtubeVideoId || (isYouTubeLibraryVideo && video.path ? video.path.replace('__youtube__', '') : null);
    
    // For local videos, check for active jobs
    if (!isYouTubeLibraryVideo && video.path) {
        const activeJob = getYouTubeJobByVideoPath(video.path, currentYouTubeAccountId || youtubeActiveAccountId || null);
        if (activeJob && activeJob.status && activeJob.status !== 'failed' && activeJob.status !== 'completed') {
            alert('An upload or update is already in progress for this video.');
            return;
        }
    }

    ensureYouTubePrivacyOptions();
    
    // Get existing video record
    let existingVideo = null;
    if (isYouTubeLibraryVideo) {
        // For YouTube library videos, get metadata from the library
        const libraryVideo = video.libraryVideo || getYouTubeLibraryVideo(videoId);
        if (libraryVideo) {
            existingVideo = {
                videoId: libraryVideo.videoId,
                title: libraryVideo.title,
                description: libraryVideo.description || '',
                privacy: libraryVideo.privacyStatus || 'unlisted',
                embeddable: libraryVideo.embeddable !== undefined ? libraryVideo.embeddable : true,
                accountId: currentYouTubeAccountId || youtubeActiveAccountId || null
            };
        } else if (videoId) {
            // Fallback: create minimal record from videoId
            existingVideo = {
                videoId: videoId,
                title: video.name || 'YouTube Video',
                description: '',
                privacy: 'unlisted',
                embeddable: true, // Default to true
                accountId: currentYouTubeAccountId || youtubeActiveAccountId || null
            };
        }
    } else {
        // For local videos, get the YouTube record
        existingVideo = getYouTubeRecord(video.path);
        // Ensure embeddable is set (default to true if not in record)
        if (existingVideo && existingVideo.embeddable === undefined) {
            existingVideo.embeddable = true;
        }
    }
    
    youtubeUploadContext = {
        video,
        mode,
        existingVideo,
        submitting: false,
        accountId: null,
        videoId: videoId || (existingVideo && existingVideo.videoId) || null,
        isYouTubeLibraryVideo: isYouTubeLibraryVideo
    };

    const defaultAccountId = (youtubeUploadContext.existingVideo && youtubeUploadContext.existingVideo.accountId)
        || currentYouTubeAccountId
        || youtubeActiveAccountId
        || (youtubeAccounts[0] && youtubeAccounts[0].id)
        || null;
    youtubeUploadContext.accountId = defaultAccountId;
    ensureYouTubeAccountOptions(defaultAccountId);

    populateYouTubeUploadForm(video, mode, youtubeUploadContext.existingVideo);
    setYouTubeUploadError('');

    if (youtubeUploadModal) {
        youtubeUploadModal.classList.add('active');
    }

    setTimeout(() => {
        if (youtubeUploadTitleInput) {
            youtubeUploadTitleInput.focus();
            youtubeUploadTitleInput.select();
        }
    }, 0);
}

function extractYouTubeVideoId(input) {
    if (!input || typeof input !== 'string') {
        return null;
    }
    const trimmed = input.trim();
    if (/^[a-zA-Z0-9_-]{11}$/.test(trimmed)) {
        return trimmed;
    }
    let parsedUrl;
    try {
        parsedUrl = new URL(trimmed);
    } catch {
        return null;
    }
    const host = parsedUrl.hostname.replace(/^(www\.|m\.)/, '');
    const pathSegments = parsedUrl.pathname.split('/').filter(Boolean);

    if (host === 'youtu.be' && pathSegments.length > 0) {
        return pathSegments[0];
    }

    if (host === 'youtube.com' || host === 'youtube-nocookie.com' || host === 'music.youtube.com' || host === 'gaming.youtube.com') {
        if (parsedUrl.searchParams.has('v')) {
            return parsedUrl.searchParams.get('v');
        }
        if (pathSegments.length >= 2 && (pathSegments[0] === 'embed' || pathSegments[0] === 'shorts' || pathSegments[0] === 'live')) {
            return pathSegments[1];
        }
    }

    if (host === 'studio.youtube.com') {
        const videoIndex = pathSegments.indexOf('video');
        if (videoIndex !== -1 && pathSegments.length > videoIndex + 1) {
            return pathSegments[videoIndex + 1];
        }
    }

    if (host.endsWith('youtube.com') && parsedUrl.searchParams.has('v')) {
        return parsedUrl.searchParams.get('v');
    }

    return null;
}

function buildYouTubeEmbedUrl(videoId) {
    if (!videoId) {
        return '';
    }
    // For Electron file:// protocol, don't set origin - let YouTube handle it
    // Setting origin to file:// can cause postMessage issues
    const params = new URLSearchParams({
        rel: '0',
        modestbranding: '1',
        playsinline: '1',
        enablejsapi: '1'
        // Note: origin parameter is not needed and can cause issues in Electron
    });
    return `https://www.youtube-nocookie.com/embed/${videoId}?${params.toString()}`;
}

function resetYouTubePlayerModal() {
    // Don't reset iframe src - it causes it to load index.html
    // Just clear the state, the iframe will be set when opening a new video
    if (youtubePlayerFrame) {
        youtubePlayerFrame.removeAttribute('data-error');
    }
    if (youtubePlayerTitle) {
        youtubePlayerTitle.textContent = 'YouTube Player';
    }
    youtubePlayerCurrentUrl = null;
    youtubePlayerCurrentVideoId = null;
    youtubePlayerHadError = false;
    if (youtubePlayerMessage) {
        youtubePlayerMessage.textContent = '';
        youtubePlayerMessage.classList.remove('visible', 'error');
    }
    if (youtubePlayerEditBtn) {
        youtubePlayerEditBtn.style.display = 'none';
    }
}

function setYouTubePlayerMessage(message, { isError = false } = {}) {
    if (!youtubePlayerMessage) {
        return;
    }
    youtubePlayerMessage.textContent = message || '';
    if (message) {
        youtubePlayerMessage.classList.add('visible');
        if (isError) {
            youtubePlayerMessage.classList.add('error');
        } else {
            youtubePlayerMessage.classList.remove('error');
        }
    } else {
        youtubePlayerMessage.classList.remove('visible');
        youtubePlayerMessage.classList.remove('error');
    }
}

function closeYouTubePlayerModal({ openExternally = false } = {}) {
    if (!youtubePlayerModal) {
        return;
    }
    youtubePlayerModal.classList.remove('active');
    youtubePlayerModal.setAttribute('aria-hidden', 'true');
    const externalUrl = youtubePlayerCurrentUrl;
    resetYouTubePlayerModal();
    if (openExternally && externalUrl) {
        try {
            window.open(externalUrl, '_blank', 'noopener');
        } catch (error) {
            logger.warn('Failed to open YouTube link in browser:', error);
        }
        return;
    }
}

// ============================================================================
// YouTube Library Video Functions (Pure YouTube, no local file)
// ============================================================================

/**
 * Get a YouTube library video by videoId
 * Returns the video object from the YouTube library (no local file)
 */
function getYouTubeLibraryVideo(videoId) {
    if (!videoId || !youtubeLibrary || !Array.isArray(youtubeLibrary.videos)) {
        return null;
    }
    return youtubeLibrary.videos.find(v => v && v.videoId === videoId) || null;
}

/**
 * Check if a videoId exists in the YouTube library
 */
function isVideoInYouTubeLibrary(videoId) {
    if (!videoId) {
        return false;
    }
    return Boolean(getYouTubeLibraryVideo(videoId));
}

/**
 * Create a video object for editing YouTube library videos (no local file)
 */
function createYouTubeLibraryVideoObject(videoId, libraryVideo = null) {
    if (!videoId) {
        return null;
    }
    
    const video = libraryVideo || getYouTubeLibraryVideo(videoId);
    if (!video) {
        // Fallback: create minimal object from videoId and title if available
        const title = youtubePlayerTitle ? youtubePlayerTitle.textContent : 'YouTube Video';
        return {
            path: `__youtube__${videoId}`,
            name: title,
            youtubeVideoId: videoId,
            isYouTubeLibraryVideo: true
        };
    }
    
    return {
        path: `__youtube__${videoId}`,
        name: video.title || 'YouTube Video',
        youtubeVideoId: videoId,
        isYouTubeLibraryVideo: true,
        libraryVideo: video
    };
}

// ============================================================================
// Local Video with YouTube Record Functions
// ============================================================================

/**
 * Find a local video file that has been uploaded to YouTube (has a local path)
 */
function findLocalVideoByYouTubeVideoId(videoId) {
    if (!videoId || !youtubeState.videos) {
        return null;
    }
    
    // Search through local YouTube records to find one matching the videoId
    for (const [videoPath, record] of youtubeState.videos.entries()) {
        if (record && record.videoId === videoId) {
            const localVideo = videoMap.get(videoPath);
            if (localVideo) {
                return localVideo;
            }
        }
    }
    
    return null;
}

function openYouTubeEmbeddedPlayer({ title, url, videoId }) {
    // Extract videoId - prioritize parameter, then extract from URL
    let resolvedId = null;
    
    if (videoId && typeof videoId === 'string') {
        const trimmed = videoId.trim();
        if (/^[a-zA-Z0-9_-]{11}$/.test(trimmed)) {
            resolvedId = trimmed;
        } else {
            resolvedId = extractYouTubeVideoId(trimmed);
        }
    }
    
    if (!resolvedId && url) {
        resolvedId = extractYouTubeVideoId(url);
    }
    
    // Double-check: get videoId from library if available
    if (resolvedId) {
        const libraryVideo = getYouTubeLibraryVideo(resolvedId);
        if (libraryVideo && libraryVideo.videoId) {
            const libId = String(libraryVideo.videoId).trim();
            if (libId && libId.length === 11) {
                resolvedId = libId;
            }
        }
    }
    
    if (!resolvedId || resolvedId.length !== 11) {
        if (url) {
            window.open(url, '_blank', 'noopener');
        }
        return;
    }
    
    // Build embed URL
    const embedUrl = buildYouTubeEmbedUrl(resolvedId);
    if (!embedUrl || !embedUrl.includes(resolvedId)) {
        return;
    }
    
    // Update state FIRST
    youtubePlayerCurrentVideoId = resolvedId;
    youtubePlayerCurrentUrl = url || `https://youtu.be/${resolvedId}`;
    
    // Show modal FIRST so iframe is in DOM
    if (youtubePlayerModal) {
        youtubePlayerModal.classList.add('active');
        youtubePlayerModal.setAttribute('aria-hidden', 'false');
    }
    
    // Update UI
    if (youtubePlayerTitle) {
        youtubePlayerTitle.textContent = title || 'YouTube Player';
    }
    
    if (youtubePlayerMessage) {
        youtubePlayerMessage.textContent = '';
        youtubePlayerMessage.classList.remove('visible', 'error');
    }
    
    if (youtubePlayerEditBtn) {
        youtubePlayerEditBtn.style.display = 'inline-block';
        youtubePlayerEditBtn.disabled = false;
    }
    
    // Set iframe src directly - no caching, no resets, just set it
    if (youtubePlayerFrame) {
        youtubePlayerFrame.src = embedUrl;
    }
    
    // Focus close button
    if (youtubePlayerCloseBtn) {
        setTimeout(() => {
            try {
                youtubePlayerCloseBtn.focus();
            } catch {
                // Ignore
            }
        }, 0);
    }
}

const YOUTUBE_PLAYER_ORIGIN_REGEX = /^https:\/\/(\w+\.)?(youtube\.com|youtube-nocookie\.com)$/i;

function handleYouTubePlayerIframeMessage(event) {
    if (!youtubePlayerModal || !youtubePlayerModal.classList.contains('active')) {
        return;
    }
    if (!YOUTUBE_PLAYER_ORIGIN_REGEX.test(event.origin)) {
        return;
    }
    if (!event || typeof event.data === 'undefined') {
        return;
    }
    let payload = event.data;
    if (typeof payload === 'string') {
        try {
            payload = JSON.parse(payload);
        } catch {
            return;
        }
    }
    if (!payload || typeof payload !== 'object') {
        return;
    }
    
    if (payload.event === 'onReady') {
        youtubePlayerHadError = false;
        setYouTubePlayerMessage('');
        return;
    }
    if (payload.event === 'onError') {
        const errorCode = typeof payload.info === 'number' ? payload.info : null;
        youtubePlayerHadError = true;
        const friendlyMessage = errorCode
            ? `YouTube reported error ${errorCode}. This video's publisher has disabled embedding.`
            : 'This video cannot be embedded.';
        const externalUrl = youtubePlayerCurrentUrl;
        if (externalUrl) {
            setYouTubePlayerMessage(`${friendlyMessage} We'll open it in your browser.`, { isError: true });
            setTimeout(() => {
                try {
                    window.open(externalUrl, '_blank', 'noopener');
                } catch (error) {
                    logger.warn('Failed to open YouTube link in browser:', error);
                    setYouTubePlayerMessage(`${friendlyMessage} Use "Open in YouTube" to watch it.`, { isError: true });
                    if (youtubePlayerOpenBtn) {
                        youtubePlayerOpenBtn.focus();
                    }
                    return;
                }
                closeYouTubePlayerModal();
            }, 100);
        } else {
            setYouTubePlayerMessage(`${friendlyMessage} Use "Open in YouTube" to watch it.`, { isError: true });
            if (youtubePlayerOpenBtn) {
                youtubePlayerOpenBtn.focus();
            }
        }
    }
}

if (typeof window !== 'undefined' && typeof window.addEventListener === 'function') {
    window.addEventListener('message', handleYouTubePlayerIframeMessage);
}

function renderYouTubeDetail(video) {
    const container = document.getElementById('youtubeDetailsContent');
    if (!container) {
        return;
    }

    container.innerHTML = '';

    if (!video) {
        container.textContent = 'Select a video to manage YouTube uploads.';
        return;
    }

    const record = getYouTubeRecord(video.path);
    const job = getYouTubeJobByVideoPath(video.path, currentYouTubeAccountId || youtubeActiveAccountId || null);
    const isUploaded = Boolean(record && record.videoId);
    const isRemoving = Boolean(record && record.status === 'removing');
    const jobActive = Boolean(job && job.status && job.status !== 'failed' && job.status !== 'completed');

    if (jobActive) {
        const progressWrapper = document.createElement('div');
        progressWrapper.className = 'youtube-upload-progress';

        const statusLabel = document.createElement('div');
        statusLabel.className = 'youtube-upload-status';
        const percentage = typeof job.progress === 'number' ? Math.max(0, Math.min(100, job.progress)) : null;
        const statusText = job.status === 'processing' ? 'Processing on YouTube…' : 'Uploading to YouTube…';
        statusLabel.textContent = percentage !== null ? `${statusText} ${percentage.toFixed(1)}%` : statusText;
        progressWrapper.appendChild(statusLabel);

        const progressBar = document.createElement('div');
        progressBar.className = 'youtube-upload-progress-bar';
        const progressFill = document.createElement('div');
        progressFill.className = 'youtube-upload-progress-fill';
        progressFill.style.width = percentage !== null ? `${percentage}%` : '15%';
        progressBar.appendChild(progressFill);
        progressWrapper.appendChild(progressBar);

        container.appendChild(progressWrapper);
    } else if (job && job.status === 'failed') {
        const errorMessage = document.createElement('div');
        errorMessage.className = 'youtube-upload-error';
        errorMessage.textContent = job.error || 'Upload failed.';
        container.appendChild(errorMessage);
    }

    if (isRemoving) {
        const removingMessage = document.createElement('div');
        removingMessage.className = 'youtube-upload-status';
        removingMessage.textContent = 'Removing video from YouTube…';
        container.appendChild(removingMessage);
    }

    if (record && record.lastError && (!job || job.status === 'failed' || job.status === 'completed')) {
        const errorMessage = document.createElement('div');
        errorMessage.className = 'youtube-upload-error';
        errorMessage.textContent = record.lastError;
        container.appendChild(errorMessage);
    }

    if (isUploaded) {
        const infoList = document.createElement('div');
        infoList.className = 'youtube-upload-info';

        const titleRow = document.createElement('div');
        titleRow.className = 'youtube-upload-info-row';
        titleRow.innerHTML = `<span class="label">Title</span><span>${record.title || video.name}</span>`;
        infoList.appendChild(titleRow);

        const privacyRow = document.createElement('div');
        privacyRow.className = 'youtube-upload-info-row';
        privacyRow.innerHTML = `<span class="label">Visibility</span><span>${(record.privacy || 'unknown').toUpperCase()}</span>`;
        infoList.appendChild(privacyRow);

        if (record.accountId) {
            const accountRow = document.createElement('div');
            accountRow.className = 'youtube-upload-info-row';
            const account = youtubeAccounts.find(acc => acc.id === record.accountId);
            const accountName = account ? getYouTubeAccountLabel(account) : record.accountId;
            accountRow.innerHTML = `<span class="label">Account</span><span>${escapeHtml(accountName)}</span>`;
            infoList.appendChild(accountRow);
        }

        if (record.publishedAt) {
            const publishedRow = document.createElement('div');
            publishedRow.className = 'youtube-upload-info-row';
            const publishedDate = new Date(record.publishedAt);
            publishedRow.innerHTML = `<span class="label">Published</span><span>${publishedDate.toLocaleString()}</span>`;
            infoList.appendChild(publishedRow);
        }

        if (record.videoId) {
            const idRow = document.createElement('div');
            idRow.className = 'youtube-upload-info-row';
            idRow.innerHTML = `<span class="label">Video ID</span><span>${record.videoId}</span>`;
            infoList.appendChild(idRow);
        }

        container.appendChild(infoList);
    } else if (!job || job.status === 'failed') {
        const placeholder = document.createElement('div');
        placeholder.className = 'youtube-upload-placeholder';
        placeholder.textContent = 'This video has not been uploaded to YouTube yet.';
        container.appendChild(placeholder);
    }

    const actions = document.createElement('div');
    actions.className = 'youtube-detail-actions';

    const primaryButton = document.createElement('button');
    primaryButton.className = 'btn';
    primaryButton.textContent = isUploaded ? 'Update Metadata' : 'Upload to YouTube';
    primaryButton.disabled = jobActive || isRemoving;
    primaryButton.addEventListener('click', () => {
        openYouTubeUploadModal(video, isUploaded ? 'update' : 'upload');
    });
    actions.appendChild(primaryButton);

    if (isUploaded) {
        const viewButton = document.createElement('button');
        viewButton.className = 'btn btn-secondary';
            viewButton.textContent = 'Watch';
        viewButton.addEventListener('click', () => {
            const videoUrl = record.url || `https://youtu.be/${record.videoId}`;
                openYouTubeEmbeddedPlayer({
                    title: record.title || video.name,
                    url: videoUrl,
                    videoId: record.videoId
                });
        });
        actions.appendChild(viewButton);

        const removeButton = document.createElement('button');
        removeButton.className = 'btn btn-secondary';
        removeButton.textContent = 'Remove from YouTube';
        removeButton.disabled = jobActive || isRemoving;
        removeButton.addEventListener('click', () => {
            removeYouTubeVideo(video);
        });
        actions.appendChild(removeButton);
    }

    if (actions.childElementCount > 0) {
        container.appendChild(actions);
    }
}

async function submitYouTubeUpload(event) {
    if (event) {
        event.preventDefault();
    }

    if (!youtubeUploadContext || !youtubeUploadContext.video) {
        setYouTubeUploadError('No video selected for upload.');
        return;
    }

    if (youtubeUploadContext.submitting) {
        return;
    }

    const title = youtubeUploadTitleInput ? youtubeUploadTitleInput.value.trim() : '';
    if (!title) {
        setYouTubeUploadError('Title is required.');
        if (youtubeUploadTitleInput) {
            youtubeUploadTitleInput.focus();
        }
        return;
    }

    const description = youtubeUploadDescriptionInput ? youtubeUploadDescriptionInput.value.trim() : '';
    const privacy = youtubeUploadPrivacySelect ? youtubeUploadPrivacySelect.value : 'unlisted';
    const embeddable = youtubeUploadEmbeddableCheckbox ? youtubeUploadEmbeddableCheckbox.checked : true;

    if (!youtubeUploadContext.accountId) {
        setYouTubeUploadError('Choose an active YouTube account before uploading.');
        return;
    }

    youtubeUploadContext.submitting = true;
    setYouTubeUploadError('');

    if (youtubeUploadSubmitBtn) {
        youtubeUploadSubmitBtn.disabled = true;
        youtubeUploadSubmitBtn.textContent = youtubeUploadContext.mode === 'update' ? 'Saving…' : 'Starting upload…';
    }

    const baseEventProps = {
        mode: youtubeUploadContext.mode,
        privacy,
        title_length: Math.min(title.length, 120),
        has_description: description ? 1 : 0
    };
    trackEvent(
        youtubeUploadContext.mode === 'upload' ? 'youtube_upload_started' : 'youtube_metadata_update_started',
        baseEventProps
    );

    try {
        const submittedVideo = youtubeUploadContext.video;
        if (youtubeUploadContext.mode === 'upload') {
            const response = await ipcRenderer.invoke('youtube.upload.start', {
                videoPath: submittedVideo.path,
                metadata: {
                    title,
                    description,
                    privacy
                },
                accountId: youtubeUploadContext.accountId
            });

            if (!response || !response.success) {
                throw new Error(response && response.error ? response.error : 'Failed to start YouTube upload.');
            }

            if (response.job) {
                const job = upsertYouTubeJob(response.job);
                if (job && job.videoPath) {
                    updateYouTubeRecord(job.videoPath, {
                        status: job.status || 'uploading',
                        lastError: null
                    });
                }
            } else {
                updateYouTubeRecord(submittedVideo.path, {
                    status: 'uploading',
                    lastError: null
                });
            }
            trackEvent('youtube_upload_queued', {
                ...baseEventProps,
                has_job: response && response.job ? 1 : 0
            });
        } else {
            // Update metadata - works for both local videos and YouTube library videos
            const isYouTubeLibraryVideo = youtubeUploadContext.isYouTubeLibraryVideo || false;
            const videoId = youtubeUploadContext.videoId 
                || (youtubeUploadContext.existingVideo && youtubeUploadContext.existingVideo.videoId)
                || (submittedVideo.youtubeVideoId)
                || undefined;
            
            if (!videoId) {
                throw new Error('Video ID is required to update metadata.');
            }
            
            const response = await ipcRenderer.invoke('youtube.video.updateMetadata', {
                videoPath: isYouTubeLibraryVideo ? undefined : submittedVideo.path,
                videoId: videoId,
                metadata: {
                    title,
                    description,
                    privacy,
                    embeddable: embeddable
                }
            });

            if (!response || !response.success) {
                throw new Error(response && response.error ? response.error : 'Failed to update YouTube metadata.');
            }

            // Only update local record if we have a real path (local videos only)
            if (!isYouTubeLibraryVideo && submittedVideo.path) {
                if (response.video) {
                    updateYouTubeRecord(submittedVideo.path, {
                        ...response.video,
                        lastError: null
                    });
                } else {
                    updateYouTubeRecord(submittedVideo.path, {
                        title,
                        description,
                        privacy,
                        lastError: null
                    });
                }
            }
            trackEvent('youtube_metadata_updated', baseEventProps);
        }

        youtubeUploadContext.submitting = false;
        
        // Get preserved state BEFORE closing modal
        const preservedState = youtubeUploadContext.preservedPlayerState;
        const wasYouTubeLibraryVideo = preservedState && preservedState.videoId;
        
        closeYouTubeUploadModal();

        if (currentVideo && submittedVideo && currentVideo.path === submittedVideo.path) {
            renderYouTubeDetail(currentVideo);
        }
        
        // For YouTube library videos, re-open the player with updated metadata
        // Wait a bit for the event handler to update the library cache
        if (wasYouTubeLibraryVideo && preservedState && preservedState.videoId) {
            // Ensure we have a clean state before re-opening
            // Close the modal completely first if it's still open
            if (youtubePlayerModal && youtubePlayerModal.classList.contains('active')) {
                youtubePlayerModal.classList.remove('active');
                youtubePlayerModal.setAttribute('aria-hidden', 'true');
            }
            
            // Reset the iframe completely
            if (youtubePlayerFrame) {
                youtubePlayerFrame.src = '';
                youtubePlayerFrame.removeAttribute('data-error');
            }
            
            setTimeout(() => {
                // Get updated video info from library (may have been updated by event handler)
                const libraryVideo = getYouTubeLibraryVideo(preservedState.videoId);
                const updatedTitle = libraryVideo ? libraryVideo.title : preservedState.title;
                
                // Re-open the player so the user can continue watching/editing
                // Make sure we're using the exact videoId that was preserved
                openYouTubeEmbeddedPlayer({
                    title: updatedTitle || preservedState.title || 'YouTube Video',
                    url: preservedState.url || `https://youtu.be/${preservedState.videoId}`,
                    videoId: preservedState.videoId
                });
            }, 300);
        }
    } catch (error) {
        setYouTubeUploadError(error && error.message ? error.message : 'Failed to communicate with YouTube.');
        youtubeUploadContext.submitting = false;
        if (youtubeUploadSubmitBtn) {
            youtubeUploadSubmitBtn.disabled = false;
            youtubeUploadSubmitBtn.textContent = youtubeUploadContext.mode === 'update' ? 'Save Changes' : 'Start Upload';
        }
        trackEvent(
            youtubeUploadContext.mode === 'upload' ? 'youtube_upload_failed' : 'youtube_metadata_update_failed',
            {
                ...baseEventProps,
                error_message: error && error.message ? error.message : 'Unknown error'
            }
        );
    }
}

async function removeYouTubeVideo(video) {
    if (!video) {
        return;
    }

    const record = getYouTubeRecord(video.path);
    if (!record || !record.videoId) {
        return;
    }

    const confirmed = confirm(`Remove "${record.title || video.name}" from YouTube?`);
    if (!confirmed) {
        return;
    }

    updateYouTubeRecord(video.path, {
        status: 'removing',
        lastError: null
    });
    renderYouTubeDetail(video);

    try {
        const response = await ipcRenderer.invoke('youtube.video.remove', {
            videoPath: video.path,
            videoId: record.videoId
        });

        if (!response || !response.success) {
            throw new Error(response && response.error ? response.error : 'Failed to remove video from YouTube.');
        }

        removeYouTubeRecord(video.path);
        youtubeState.jobsByVideoPath.delete(video.path);

        if (currentVideo && currentVideo.path === video.path) {
            renderYouTubeDetail(currentVideo);
        }
        trackEvent('youtube_video_removed', getVideoAnalyticsProps(video));
    } catch (error) {
        updateYouTubeRecord(video.path, {
            status: 'uploaded',
            lastError: error && error.message ? error.message : 'Failed to remove video from YouTube.'
        });
        if (currentVideo && currentVideo.path === video.path) {
            renderYouTubeDetail(currentVideo);
        }
        trackEvent('youtube_video_remove_failed', {
            ...getVideoAnalyticsProps(video),
            error_message: error && error.message ? error.message : 'Unknown error'
        });
    }
}

function handleYouTubeUploadProgress(payload) {
    if (!payload || !payload.jobId || !payload.videoPath) {
        return;
    }

    const accountId = payload.accountId || null;

    const job = upsertYouTubeJob({
        id: payload.jobId,
        videoPath: payload.videoPath,
        accountId,
        progress: payload.progress,
        status: payload.status || 'uploading',
        error: payload.error || null
    });

    if (payload.uploadedVideo) {
        updateYouTubeRecord(payload.videoPath, {
            ...payload.uploadedVideo,
            accountId: payload.uploadedVideo.accountId || accountId || null,
            status: job.status,
            lastError: null
        });
    } else {
        const update = {
            status: job.status,
            lastError: null
        };
        if (accountId) {
            update.accountId = accountId;
        }
        updateYouTubeRecord(payload.videoPath, update);
    }

    if (currentVideo && currentVideo.path === payload.videoPath) {
        renderYouTubeDetail(currentVideo);
    }

    if (viewMode === 'gallery') {
        const now = Date.now();
        if (!youtubeState.lastGalleryRefresh || now - youtubeState.lastGalleryRefresh > 800) {
            youtubeState.lastGalleryRefresh = now;
            renderGallery();
        }
    }
}

function handleYouTubeUploadCompleted(payload) {
    if (!payload || !payload.jobId || !payload.videoPath) {
        return;
    }

    const accountId = payload.accountId || null;

    const status = payload.success ? 'completed' : 'failed';
    upsertYouTubeJob({
        id: payload.jobId,
        videoPath: payload.videoPath,
        accountId,
        progress: payload.success ? 100 : payload.progress || 0,
        status,
        error: payload.error || null
    });

    if (payload.success) {
        if (payload.uploadedVideo) {
            updateYouTubeRecord(payload.videoPath, {
                ...payload.uploadedVideo,
                accountId: payload.uploadedVideo.accountId || accountId || null,
                status: 'uploaded',
                lastError: null
            });
        } else {
            const update = {
                status: 'uploaded',
                lastError: null
            };
            if (accountId) {
                update.accountId = accountId;
            }
            updateYouTubeRecord(payload.videoPath, update);
        }
    } else {
        const update = {
            status: 'error',
            lastError: payload.error || 'YouTube upload failed.'
        };
        if (accountId) {
            update.accountId = accountId;
        }
        updateYouTubeRecord(payload.videoPath, update);
    }

    const jobKey = buildYouTubeJobKey(payload.videoPath, accountId);
    youtubeState.jobsByVideoPath.delete(jobKey);

    if (currentVideo && currentVideo.path === payload.videoPath) {
        renderYouTubeDetail(currentVideo);
    }

    if (viewMode === 'gallery') {
        youtubeState.lastGalleryRefresh = Date.now();
        renderGallery();
    } else {
        youtubeState.lastGalleryRefresh = Date.now();
    }

    if (payload.success && accountId && payload.uploadedVideo) {
        // Add uploaded video to cache immediately without full reload
        const uploadedVideo = payload.uploadedVideo;
        const videoData = {
            videoId: uploadedVideo.videoId,
            title: uploadedVideo.title || 'Untitled Video',
            description: uploadedVideo.description || '',
            privacy: uploadedVideo.privacy || 'unlisted',
            privacyStatus: uploadedVideo.privacy || 'unlisted',
            publishedAt: uploadedVideo.publishedAt || new Date().toISOString(),
            url: uploadedVideo.url || (uploadedVideo.videoId ? `https://youtu.be/${uploadedVideo.videoId}` : null),
            accountId: accountId,
            embeddable: uploadedVideo.embeddable !== undefined ? uploadedVideo.embeddable : true
        };

        // Add to cache via IPC
        ipcRenderer.invoke('youtube.cache.addVideo', {
            accountId: accountId,
            video: videoData
        }).catch((error) => {
            logger.warn('Failed to add video to cache:', error && error.message ? error.message : error);
        });

        // Update local library cache immediately
        const library = ensureYouTubeLibraryCache(accountId);
        if (Array.isArray(library.videos)) {
            // Check if video already exists
            const existingIndex = library.videos.findIndex(v => v && v.videoId === videoData.videoId);
            const normalizedVideo = {
                ...videoData,
                thumbnails: {},
                thumbnailUrl: null,
                channelTitle: library.channelTitle || '',
                channelId: library.channelId || null,
                durationSeconds: 0,
                liveBroadcastContent: 'none',
                status: 'uploaded',
                type: 'videos',
                playlistIds: []
            };
            
            if (existingIndex >= 0) {
                library.videos[existingIndex] = normalizedVideo;
            } else {
                library.videos.unshift(normalizedVideo);
            }
            
            library.fetchedAt = Date.now();
            youtubeLibraryByAccount.set(accountId, library);
            youtubeState.libraryByAccount = youtubeLibraryByAccount;
            state.youtube.libraryByAccount = youtubeLibraryByAccount;
            
            if (accountId === currentYouTubeAccountId || accountId === youtubeActiveAccountId) {
                youtubeLibrary = library;
                youtubeState.library = youtubeLibrary;
                state.youtube.library = youtubeLibrary;
                resetYouTubeLibraryIndex(accountId);
            }
            
            // Refresh gallery if viewing YouTube content
            if (isYouTubeTreeKey(currentTreeSelection)) {
                filterAndRenderGallery(currentTreeSelection);
            }
        }
    }
}

async function hydrateYouTubeState(options = {}) {
    if (!ipcRenderer) {
        return;
    }

    logger.log('[Renderer] hydrateYouTubeState called with options:', JSON.stringify(options));
    try {
        // Get the current active accountId to pass to the IPC handler
        const activeAccountId = currentYouTubeAccountId || youtubeActiveAccountId || null;
        
        const responses = await Promise.allSettled([
            ipcRenderer.invoke('youtube.video.list', { accountId: activeAccountId }),
            ipcRenderer.invoke('youtube.accounts.list')
        ]);

        const recordsResult = responses[0];
        if (recordsResult.status === 'fulfilled') {
            const response = recordsResult.value;
            if (response && response.success && response.videos) {
                const videosMap = new Map();
                for (const [videoPath, record] of Object.entries(response.videos)) {
                    if (record && typeof record === 'object') {
                        videosMap.set(videoPath, record);
                    }
                }

                // Clear old records and set new ones (important when switching accounts)
                // Clear the existing map and repopulate it (don't replace the Map object)
                youtubeState.videos.clear();
                for (const [videoPath, record] of videosMap.entries()) {
                    youtubeState.videos.set(videoPath, record);
                }
                state.youtube.videos = youtubeState.videos;

                if (currentVideo) {
                    renderYouTubeDetail(currentVideo);
                }

                if (viewMode === 'gallery' && !isYouTubeTreeKey(currentTreeSelection || '')) {
                    await renderGallery();
                    youtubeState.lastGalleryRefresh = Date.now();
                }
            }
        } else if (recordsResult.status === 'rejected') {
            logger.warn('Failed to load YouTube records:', recordsResult.reason);
        }

        const accountsResult = responses[1];
        let accountsStatus = null;
        if (accountsResult.status === 'fulfilled') {
            const response = accountsResult.value;
            if (response && response.success) {
                accountsStatus = response.status || {};
            } else if (response && response.error) {
                logger.warn('Failed to load YouTube accounts:', response.error);
            }
        } else if (accountsResult.status === 'rejected') {
            logger.warn('Failed to load YouTube accounts:', accountsResult.reason);
        }

        if (accountsStatus) {
            youtubeAccounts = Array.isArray(accountsStatus.accounts) ? accountsStatus.accounts.slice() : [];
            youtubeState.accounts = youtubeAccounts;
            state.youtube.accounts = youtubeAccounts;

            youtubeActiveAccountId = accountsStatus.activeAccountId || (youtubeAccounts[0] && youtubeAccounts[0].id) || null;
            youtubeState.activeAccountId = youtubeActiveAccountId;
            state.youtube.activeAccountId = youtubeActiveAccountId;

            const knownAccountIds = new Set(youtubeAccounts.map(account => account.id));
            for (const key of Array.from(youtubeLibraryByAccount.keys())) {
                if (!knownAccountIds.has(key)) {
                    youtubeLibraryByAccount.delete(key);
                    youtubeLibraryIndexByAccount.delete(key);
                    youtubeLibraryLoadingAccounts.delete(key);
                }
            }

            youtubeAccounts.forEach(account => {
                const library = ensureYouTubeLibraryCache(account.id);
                library.accountId = account.id;
            });

            if (!currentYouTubeAccountId || !knownAccountIds.has(currentYouTubeAccountId)) {
                currentYouTubeAccountId = youtubeActiveAccountId;
                state.currentYouTubeAccountId = currentYouTubeAccountId;
            }
        } else {
            youtubeAccounts = [];
            youtubeActiveAccountId = null;
            youtubeState.accounts = [];
            youtubeState.activeAccountId = null;
            state.youtube.accounts = [];
            state.youtube.activeAccountId = null;
            youtubeLibraryByAccount.clear();
            youtubeLibraryIndexByAccount.clear();
            youtubeLibraryLoadingAccounts.clear();
            currentYouTubeAccountId = null;
            state.currentYouTubeAccountId = null;
        }

        setCurrentYouTubeAccount(currentYouTubeAccountId || youtubeActiveAccountId || null);
        if (isYouTubeTreeKey(currentTreeSelection)) {
            const parsedSelection = parseTreeSelection(currentTreeSelection);
            if (parsedSelection.accountId && parsedSelection.accountId !== currentYouTubeAccountId) {
                let nextSelection = null;
                if (parsedSelection.type === 'playlist' && parsedSelection.playlistId) {
                    nextSelection = buildYouTubePlaylistKey(currentYouTubeAccountId, parsedSelection.playlistId);
                } else if (parsedSelection.category) {
                    nextSelection = buildYouTubeCategoryKey(currentYouTubeAccountId, parsedSelection.category);
                } else {
                    nextSelection = YOUTUBE_ROOT_KEY;
                }
                currentTreeSelection = nextSelection;
                state.currentTreeSelection = currentTreeSelection;
            }
        }
        ensureYouTubeAccountOptions(youtubeUploadContext && youtubeUploadContext.accountId ? youtubeUploadContext.accountId : null);

        const shouldLoadLibrary = Boolean(youtubeAuthState && youtubeAuthState.status && youtubeAuthState.status.authenticated && youtubeAccounts.length > 0);
        logger.log('[Renderer] hydrateYouTubeState: shouldLoadLibrary=', shouldLoadLibrary, 'options.force=', options.force);
        if (shouldLoadLibrary) {
            const targetAccountId = currentYouTubeAccountId || youtubeActiveAccountId || youtubeAccounts[0].id;
            logger.log('[Renderer] hydrateYouTubeState: targetAccountId=', targetAccountId);
            
            // Try to load cache first
            let cacheLoaded = false;
            if (!options.force) {
                logger.log('[Renderer] hydrateYouTubeState: Attempting to load cache for account:', targetAccountId);
                // Set cache loading flag (don't trigger gallery update here to avoid loops)
                youtubeLibraryCacheLoadingAccounts.set(targetAccountId, true);
                
                try {
                    const cacheResponse = await ipcRenderer.invoke('youtube.cache.load', { accountId: targetAccountId });
                    logger.log('[Renderer] hydrateYouTubeState: cacheResponse.success=', cacheResponse && cacheResponse.success, 'has cache=', !!(cacheResponse && cacheResponse.cache));
                    if (cacheResponse && cacheResponse.success && cacheResponse.cache) {
                        const cachedLibrary = ensureYouTubeLibraryCache(targetAccountId);
                        cachedLibrary.accountId = cacheResponse.cache.accountId || targetAccountId;
                        cachedLibrary.channelId = cacheResponse.cache.channelId || null;
                        cachedLibrary.channelTitle = cacheResponse.cache.channelTitle || null;
                        cachedLibrary.videos = Array.isArray(cacheResponse.cache.videos) ? cacheResponse.cache.videos : [];
                        cachedLibrary.playlists = Array.isArray(cacheResponse.cache.playlists) ? cacheResponse.cache.playlists : [];
                        cachedLibrary.playlistItems = cacheResponse.cache.playlistItems && typeof cacheResponse.cache.playlistItems === 'object' ? cacheResponse.cache.playlistItems : {};
                        cachedLibrary.fetchedAt = cacheResponse.cache.fetchedAt ? new Date(cacheResponse.cache.fetchedAt).getTime() : Date.now();
                        
                        youtubeLibraryByAccount.set(targetAccountId, cachedLibrary);
                        youtubeState.libraryByAccount = youtubeLibraryByAccount;
                        state.youtube.libraryByAccount = youtubeLibraryByAccount;
                        
                        if (targetAccountId === currentYouTubeAccountId || targetAccountId === youtubeActiveAccountId) {
                            youtubeLibrary = cachedLibrary;
                            youtubeState.library = youtubeLibrary;
                            state.youtube.library = youtubeLibrary;
                            resetYouTubeLibraryIndex(targetAccountId);
                            
                            // Ensure loading state is false when loading from cache
                            youtubeLibraryLoadingAccounts.set(targetAccountId, false);
                            youtubeLibraryCacheLoadingAccounts.set(targetAccountId, false);
                            youtubeLibraryLoading = false;
                        } else {
                            // Ensure loading state is false for other accounts too
                            youtubeLibraryLoadingAccounts.set(targetAccountId, false);
                            youtubeLibraryCacheLoadingAccounts.set(targetAccountId, false);
                        }
                        
                        cacheLoaded = true;
                    }
                } catch (error) {
                    logger.warn('Failed to load YouTube cache:', error && error.message ? error.message : error);
                } finally {
                    // Clear cache loading flag
                    youtubeLibraryCacheLoadingAccounts.set(targetAccountId, false);
                }
            }
            
            // Only fetch from YouTube if cache wasn't loaded or force is true
            logger.log('[Renderer] hydrateYouTubeState: cacheLoaded=', cacheLoaded, 'options.force=', options.force, 'targetAccountId=', targetAccountId);
            if (!cacheLoaded || options.force) {
                logger.log('[Renderer] hydrateYouTubeState: Will call loadYouTubeLibraryForAccount with force:', options.force === true);
                await loadYouTubeLibraryForAccount(targetAccountId, {
                    force: options.force === true,
                    maxVideos: options.maxVideos || 200,
                    maxPlaylists: options.maxPlaylists || 50,
                    maxPlaylistItems: options.maxPlaylistItems || 100
                });
            } else {
                logger.log('[Renderer] hydrateYouTubeState: Using cache, skipping API call');
            }
        } else {
            youtubeLibrary = createEmptyYouTubeLibrary();
            youtubeState.library = youtubeLibrary;
            state.youtube.library = youtubeLibrary;
            resetYouTubeLibraryIndex();
        }
    } catch (error) {
        logger.warn('Failed to synchronize YouTube state:', error && error.message ? error.message : error);
    } finally {
        const currentAccountKey = currentYouTubeAccountId || youtubeActiveAccountId || '__default__';
        youtubeLibraryLoading = Boolean(youtubeLibraryLoadingAccounts.get(currentAccountKey));

        const hasDirectories = Array.isArray(settings.monitoredDirectories) && settings.monitoredDirectories.length > 0;
        if (!hasDirectories && youtubeAccounts.length > 0 && !isYouTubeTreeKey(currentTreeSelection)) {
            currentTreeSelection = YOUTUBE_ROOT_KEY;
            state.currentTreeSelection = currentTreeSelection;
        }
        updateDirectoryTree();
        filterAndRenderGallery(currentTreeSelection);
    }
}

if (youtubeUploadForm) {
    youtubeUploadForm.addEventListener('submit', submitYouTubeUpload);
}

if (youtubeUploadSubmitBtn && !youtubeUploadForm) {
    youtubeUploadSubmitBtn.addEventListener('click', submitYouTubeUpload);
}

if (youtubeUploadCancelBtn) {
    youtubeUploadCancelBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeYouTubeUploadModal();
    });
}

if (youtubeUploadCloseBtn) {
    youtubeUploadCloseBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeYouTubeUploadModal();
    });
}

if (youtubeUploadModal) {
    youtubeUploadModal.addEventListener('click', (event) => {
        if (event.target === youtubeUploadModal) {
            closeYouTubeUploadModal();
        }
    });
}

if (youtubePlayerCloseBtn) {
    youtubePlayerCloseBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeYouTubePlayerModal();
    });
}

if (youtubePlayerOpenBtn) {
    youtubePlayerOpenBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeYouTubePlayerModal({ openExternally: true });
    });
}

if (youtubePlayerEditBtn) {
    youtubePlayerEditBtn.addEventListener('click', (event) => {
        event.preventDefault();
        if (!youtubePlayerCurrentVideoId) {
            return;
        }
        
        // Preserve the videoId BEFORE doing anything else
        const preservedVideoId = youtubePlayerCurrentVideoId;
        const preservedVideoUrl = youtubePlayerCurrentUrl;
        const preservedVideoTitle = youtubePlayerTitle ? youtubePlayerTitle.textContent : null;
        
        // Try to find local video first (uploaded from this app)
        let video = findLocalVideoByYouTubeVideoId(preservedVideoId);
        
        // If not a local video, treat it as a YouTube library video
        if (!video) {
            video = createYouTubeLibraryVideoObject(preservedVideoId);
        }
        
        if (!video) {
            alert('Could not find the video information.');
            return;
        }
        
        // Store the preserved state BEFORE closing the modal
        // We need to set this before closeYouTubePlayerModal() clears everything
        const preservedState = {
            videoId: preservedVideoId,
            url: preservedVideoUrl,
            title: preservedVideoTitle
        };
        
        // Close the player modal (this will reset the state)
        closeYouTubePlayerModal();
        
        // Open the upload modal in update mode
        openYouTubeUploadModal(video, 'update');
        
        // Store the preserved state in the upload context AFTER opening the modal
        // (youtubeUploadContext is set by openYouTubeUploadModal)
        if (youtubeUploadContext) {
            youtubeUploadContext.preservedPlayerState = preservedState;
        }
    });
}

if (youtubePlayerModal) {
    youtubePlayerModal.addEventListener('click', (event) => {
        if (event.target === youtubePlayerModal) {
            closeYouTubePlayerModal();
        }
    });
}

// Scan all directories
async function scanAllDirectories() {
    videos.length = 0;
    videoMap.clear();

    for (const dirObj of settings.monitoredDirectories) {
        const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
        const dirName = typeof dirObj === 'string' ? dirPath.split(/[\\/]/).pop() : dirObj.name;
        const dirVideos = await ipcRenderer.invoke('scan-directory', dirPath);
        for (const video of dirVideos) {
            if (!videoMap.has(video.path)) {
                // Add directory name to video object
                video.directoryName = dirName;
                video.directoryPath = dirPath;
                videos.push(video);
                videoMap.set(video.path, video);
            }
        }
    }

    filterAndRenderGallery(currentTreeSelection);
    updateDirectoryTree();
}

function normalizePath(p) {
    return p.replace(/\\/g, '/').toLowerCase();
}

function filterVideosByDirectory(videosToFilter, selectedPath) {
    if (!selectedPath) {
        return videosToFilter;
    }

    const normalizedSelected = normalizePath(selectedPath);

    return videosToFilter.filter(video => {
        const pathSeparator = video.path.includes('\\') ? '\\' : '/';
        const lastSeparatorIndex = video.path.lastIndexOf(pathSeparator);
        const videoDir = lastSeparatorIndex > 0 ? video.path.substring(0, lastSeparatorIndex) : video.path;
        const normalizedVideoDir = normalizePath(videoDir);
        return normalizedVideoDir === normalizedSelected || normalizedVideoDir.startsWith(normalizedSelected + '/');
    });
}

function getFilteredVideos(selectedPath = currentSelectedDirectory) {
    let filtered = videos;

    if (selectedPath) {
        filtered = filterVideosByDirectory(filtered, selectedPath);
    }

    if (searchQuery) {
        const query = searchQuery.toLowerCase();
        filtered = filtered.filter(video => {
            const nameMatches = video.name && video.name.toLowerCase().includes(query);
            const pathMatches = video.path && video.path.toLowerCase().includes(query);
            return nameMatches || pathMatches;
        });
    }

    return filtered;
}

function isYouTubeTreeKey(selectionKey) {
    if (!selectionKey || typeof selectionKey !== 'string') {
        return false;
    }
    return selectionKey === YOUTUBE_ROOT_KEY
        || selectionKey.startsWith(YOUTUBE_ACCOUNT_PREFIX)
        || selectionKey.startsWith(YOUTUBE_CATEGORY_PREFIX)
        || selectionKey.startsWith(YOUTUBE_PLAYLIST_PREFIX);
}

function parseTreeSelection(selectionKey) {
    const key = selectionKey || currentTreeSelection || LOCAL_ROOT_KEY;
    if (key === LOCAL_ROOT_KEY) {
        return {
            source: 'local',
            type: 'root',
            path: null
        };
    }
    if (isYouTubeTreeKey(key)) {
        if (key === YOUTUBE_ROOT_KEY) {
            return {
                source: 'youtube',
                type: 'root',
                accountId: null,
                category: null,
                playlistId: null
            };
        }
        if (key.startsWith(YOUTUBE_ACCOUNT_PREFIX)) {
            const accountSegment = key.slice(YOUTUBE_ACCOUNT_PREFIX.length);
            const accountId = accountSegment ? decodeKeyComponent(accountSegment) : null;
            return {
                source: 'youtube',
                type: 'account',
                accountId,
                category: null,
                playlistId: null
            };
        }
        if (key.startsWith(YOUTUBE_CATEGORY_PREFIX)) {
            const remainder = key.slice(YOUTUBE_CATEGORY_PREFIX.length);
            const separatorIndex = remainder.indexOf('__');
            const accountSegment = separatorIndex >= 0 ? remainder.slice(0, separatorIndex) : remainder;
            const categorySegment = separatorIndex >= 0 ? remainder.slice(separatorIndex + 2) : '';
            const accountId = accountSegment ? decodeKeyComponent(accountSegment) : null;
            const category = categorySegment ? decodeKeyComponent(categorySegment) : null;
            return {
                source: 'youtube',
                type: 'category',
                accountId,
                category,
                playlistId: null
            };
        }
        if (key.startsWith(YOUTUBE_PLAYLIST_PREFIX)) {
            const remainder = key.slice(YOUTUBE_PLAYLIST_PREFIX.length);
            const separatorIndex = remainder.indexOf('__');
            const accountSegment = separatorIndex >= 0 ? remainder.slice(0, separatorIndex) : remainder;
            const playlistSegment = separatorIndex >= 0 ? remainder.slice(separatorIndex + 2) : '';
            const accountId = accountSegment ? decodeKeyComponent(accountSegment) : null;
            const playlistId = playlistSegment ? decodeKeyComponent(playlistSegment) : null;
            return {
                source: 'youtube',
                type: 'playlist',
                accountId,
                category: 'playlists',
                playlistId
            };
        }
    }
    return {
        source: 'local',
        type: 'directory',
        path: key
    };
}

function getYouTubeCategoryLabel(categoryKey) {
    if (!categoryKey) {
        return 'YouTube';
    }
    const found = YOUTUBE_CATEGORIES.find(cat => cat.key === categoryKey);
    return found ? found.label : categoryKey;
}

// Filter videos based on selected directory and active filters
function filterAndRenderGallery(selectionKey = currentTreeSelection) {
    const context = parseTreeSelection(selectionKey);
    currentTreeSelection = selectionKey || context.path || (context.source === 'local' ? LOCAL_ROOT_KEY : YOUTUBE_ROOT_KEY);
    state.currentTreeSelection = currentTreeSelection;

    if (context.source === 'youtube') {
        currentSelectedDirectory = null;
        state.currentSelectedDirectory = null;
        const targetAccountId = context.accountId != null
            ? context.accountId
            : (currentYouTubeAccountId || youtubeActiveAccountId || (youtubeAccounts.length > 0 ? youtubeAccounts[0].id : null));
        setCurrentYouTubeAccount(targetAccountId);
        const derivedCategory = context.category || 'videos';
        currentYouTubeCategory = derivedCategory;
        state.currentYouTubeCategory = currentYouTubeCategory;
        currentYouTubePlaylist = context.type === 'playlist' ? context.playlistId || null : null;
        state.currentYouTubePlaylist = currentYouTubePlaylist;

        const effectiveContext = {
            ...context,
            accountId: currentYouTubeAccountId,
            category: context.category || derivedCategory,
            type: context.type === 'account' ? 'category' : context.type
        };

        renderYouTubeSelection(effectiveContext);

        if (targetAccountId) {
            const requestAccountId = targetAccountId;
            loadYouTubeLibraryForAccount(targetAccountId, {
                cacheMs: YOUTUBE_LIBRARY_CACHE_MS,
                maxVideos: 200,
                maxPlaylists: 50,
                maxPlaylistItems: 100
            }).then(() => {
                if (currentYouTubeAccountId === requestAccountId && parseTreeSelection(currentTreeSelection).accountId === requestAccountId) {
                    renderYouTubeSelection(parseTreeSelection(currentTreeSelection));
                    updateDirectoryTree();
                }
            }).catch((error) => {
                logger.warn('Failed to refresh YouTube library for account:', error && error.message ? error.message : error);
            });
        }
        return;
    }

    currentSelectedDirectory = context.path || null;
    state.currentSelectedDirectory = currentSelectedDirectory;
    currentYouTubeCategory = null;
    currentYouTubePlaylist = null;
    state.currentYouTubeCategory = currentYouTubeCategory;
    state.currentYouTubePlaylist = currentYouTubePlaylist;

    const videosToDisplay = getFilteredVideos(currentSelectedDirectory);
    renderGallery(videosToDisplay);
    updateStats(videosToDisplay);
}

function renderYouTubeSelection(context) {
    const container = document.getElementById('galleryContent');
    if (!container) {
        return;
    }

    renderYouTubeAccountSwitcher();

    if (!youtubeAuthState || !youtubeAuthState.status || !youtubeAuthState.status.authenticated || !Array.isArray(youtubeAccounts) || youtubeAccounts.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <h2>Connect YouTube</h2>
                <p>Link your YouTube account to view channel videos.</p>
            </div>
        `;
        updateYouTubeStats([], { context });
        return;
    }

    const selectedAccountId = context && context.accountId
        ? context.accountId
        : currentYouTubeAccountId || youtubeActiveAccountId || (youtubeAccounts[0] && youtubeAccounts[0].id) || null;
    const accountRecord = youtubeAccounts.find(acc => acc.id === selectedAccountId) || youtubeAccounts[0] || null;
    const accountLabel = accountRecord ? getYouTubeAccountLabel(accountRecord) : 'YouTube';

    setCurrentYouTubeAccount(accountRecord ? accountRecord.id : null);

    const library = accountRecord ? (youtubeLibraryByAccount.get(accountRecord.id) || createEmptyYouTubeLibrary()) : createEmptyYouTubeLibrary();
    youtubeLibrary = library;
    state.youtube.library = youtubeLibrary;
    const accountId = accountRecord ? accountRecord.id : '__default__';
    const isLibraryLoading = Boolean(youtubeLibraryLoadingAccounts.get(accountId));
    const isCacheLoading = Boolean(youtubeLibraryCacheLoadingAccounts.get(accountId));
    youtubeLibraryLoading = isLibraryLoading;

    if (isCacheLoading) {
        container.innerHTML = `
            <div class="empty-state">
                <h2>Loading ${escapeHtml(accountLabel)}…</h2>
                <p>Loading cached videos.</p>
            </div>
        `;
        updateYouTubeStats([], { context: { ...context, accountId: accountRecord ? accountRecord.id : null }, loading: true });
        return;
    }

    if (isLibraryLoading) {
        container.innerHTML = `
            <div class="empty-state">
                <h2>Loading ${escapeHtml(accountLabel)}…</h2>
                <p>Fetching videos from this channel.</p>
            </div>
        `;
        updateYouTubeStats([], { context: { ...context, accountId: accountRecord ? accountRecord.id : null }, loading: true });
        return;
    }

    if (library && library.errors) {
        // Strip HTML tags from error messages
        const errorText = String(library.errors).replace(/<[^>]*>/g, '');
        container.innerHTML = `
            <div class="empty-state">
                <h2>Unable to Load ${escapeHtml(accountLabel)}</h2>
                <p>${escapeHtml(errorText)}</p>
            </div>
        `;
        updateYouTubeStats([], { context: { ...context, accountId: accountRecord ? accountRecord.id : null } });
        return;
    }

    const effectiveContext = context || { source: 'youtube', type: 'root', category: 'videos' };

    if (effectiveContext.type === 'playlist') {
        const playlist = getYouTubePlaylistById(effectiveContext.playlistId, library);
        if (!playlist) {
            container.innerHTML = `
                <div class="empty-state">
                    <h2>Playlist Not Found</h2>
                    <p>This playlist may have been removed or is not accessible.</p>
                </div>
            `;
            updateYouTubeStats([], { context: { ...effectiveContext, accountId: accountRecord ? accountRecord.id : null } });
            return;
        }
        const playlistVideos = getYouTubeVideosForPlaylist(effectiveContext.playlistId, library);
        const filteredVideos = applySearchFilterToYouTubeItems(playlistVideos);
        renderYouTubeVideosView(filteredVideos, {
            heading: playlist.title || 'Playlist',
            subheading: playlist.description || '',
            playlist,
            accountLabel
        });
        updateYouTubeStats(filteredVideos, { context: { ...effectiveContext, accountId: accountRecord ? accountRecord.id : null }, playlist, accountLabel });
        return;
    }

    if (effectiveContext.type === 'category') {
        const categoryKey = (effectiveContext.category || '').toLowerCase();

        const videosByCategory = getYouTubeVideosByCategory(categoryKey || 'videos', library);
        const filteredVideos = applySearchFilterToYouTubeItems(videosByCategory);
        renderYouTubeVideosView(filteredVideos, {
            heading: getYouTubeCategoryLabel(categoryKey || 'videos'),
            category: categoryKey || 'videos',
            accountLabel
        });
        updateYouTubeStats(filteredVideos, { context: { ...effectiveContext, accountId: accountRecord ? accountRecord.id : null }, accountLabel });
        return;
    }

    // YouTube root defaults to standard videos
    const defaultVideos = getYouTubeVideosByCategory('videos', library);
    const filteredDefaultVideos = applySearchFilterToYouTubeItems(defaultVideos);
    renderYouTubeVideosView(filteredDefaultVideos, {
        heading: 'Videos',
        category: 'videos',
        accountLabel
    });
    updateYouTubeStats(filteredDefaultVideos, { context: { ...effectiveContext, accountId: accountRecord ? accountRecord.id : null }, accountLabel });
}

function renderYouTubeVideosView(videosList, options = {}) {
    const container = document.getElementById('galleryContent');
    if (!container) {
        return;
    }

    if (!Array.isArray(videosList) || videosList.length === 0) {
        const heading = options.heading || 'YouTube Videos';
        
        // Check if we're loading from cache or YouTube
        const accountId = currentYouTubeAccountId || youtubeActiveAccountId || null;
        const isCacheLoading = accountId ? Boolean(youtubeLibraryCacheLoadingAccounts.get(accountId)) : false;
        const isLibraryLoading = accountId ? Boolean(youtubeLibraryLoadingAccounts.get(accountId)) : false;
        
        let emptyMessage;
        if (isCacheLoading) {
            emptyMessage = 'Loading cached content...';
        } else if (isLibraryLoading) {
            emptyMessage = 'Fetching videos from this channel.';
        } else if (searchQuery) {
            emptyMessage = 'No items match the current search.';
        } else {
            emptyMessage = 'No items are available in this section.';
        }
        
        container.innerHTML = `
            <div class="empty-state">
                <h2>${escapeHtml(heading)}</h2>
                <p>${escapeHtml(emptyMessage)}</p>
            </div>
        `;
        return;
    }

    const heading = options.heading || 'YouTube Videos';
    const accountLabel = options.accountLabel ? `<p class="section-subheading">${escapeHtml(options.accountLabel)}</p>` : '';
    const secondarySubheading = options.subheading ? `<p class="section-subheading">${escapeHtml(options.subheading)}</p>` : '';

    let html = `
        <div class="folder-section">
            <div class="folder-header">
                ${escapeHtml(heading)}
            </div>
            ${accountLabel}${secondarySubheading}
            <div class="video-grid">
    `;

    const sortedVideos = videosList.slice().sort((a, b) => {
        const posA = Number.isFinite(a && a.uploadPosition) ? a.uploadPosition : Number.MAX_SAFE_INTEGER;
        const posB = Number.isFinite(b && b.uploadPosition) ? b.uploadPosition : Number.MAX_SAFE_INTEGER;
        if (posA !== posB) {
            return posA - posB;
        }
        const timeA = a && a.publishedAt ? new Date(a.publishedAt).getTime() : 0;
        const timeB = b && b.publishedAt ? new Date(b.publishedAt).getTime() : 0;
        return timeB - timeA;
    });

    sortedVideos.forEach((video) => {
        if (!video) {
            return;
        }
        const thumbnailCandidates = video.thumbnails || {};
        const thumbnailUrl = video.thumbnailUrl
            || (thumbnailCandidates.high && thumbnailCandidates.high.url)
            || (thumbnailCandidates.medium && thumbnailCandidates.medium.url)
            || (thumbnailCandidates.default && thumbnailCandidates.default.url)
            || '';
        const durationSeconds = Number(video.durationSeconds);
        const publishedAt = video.publishedAt ? formatDate(new Date(video.publishedAt)) : null;
        const metaParts = [];
        if (Number.isFinite(durationSeconds) && durationSeconds > 0) {
            metaParts.push(formatTimestamp(durationSeconds, durationSeconds >= 3600 ? 0 : 2));
        }
        if (publishedAt) {
            metaParts.push(publishedAt);
        }
        if (video.privacy) {
            metaParts.push(String(video.privacy).toUpperCase());
        }
        const metaText = metaParts.join(' • ');
        const statusMeta = video.status ? getYouTubeStatusMeta(video, null) : null;
        // Don't show "Uploaded" status when viewing YouTube videos - they're all on YouTube already
        const statusHtml = statusMeta && statusMeta.className !== 'uploaded'
            ? `<div class="thumbnail-youtube-status ${statusMeta.className}">${escapeHtml(statusMeta.label)}</div>`
            : '';
        
        // Ensure videoId is a valid string - this is critical for playback
        const videoId = video.videoId ? String(video.videoId).trim() : '';
        if (!videoId) {
            logger.warn('Skipping YouTube video without valid videoId:', video);
            return; // Skip this video in the loop
        }
        
        const videoUrl = video.url || `https://youtu.be/${videoId}`;
        const videoTitle = video.title || 'Untitled Video';
        const videoTitleEscaped = escapeHtml(videoTitle);
        const videoAccountId = video.accountId || currentYouTubeAccountId || youtubeActiveAccountId || null;
        html += `
            <div class="video-thumbnail youtube-remote" data-video-id="${escapeHtml(videoId)}" data-url="${escapeHtml(videoUrl)}">
                ${thumbnailUrl
                    ? `<img class="thumbnail-image" src="${escapeHtml(thumbnailUrl)}" alt="">`
                    : '<div class="thumbnail-placeholder">YT</div>'}
                <div class="thumbnail-actions">
                    <button class="thumbnail-action-btn youtube-delete-btn" title="Delete from YouTube" data-video-id="${escapeHtml(videoId)}" data-video-title="${escapeHtml(videoTitle)}" data-account-id="${escapeHtml(videoAccountId || '')}">🗑</button>
                </div>
                <div class="thumbnail-info">
                    <div class="thumbnail-name">${videoTitleEscaped}</div>
                    <div class="thumbnail-meta">${escapeHtml(metaText)}</div>
                    ${statusHtml}
                </div>
            </div>
        `;
    });

    html += `
            </div>
        </div>
    `;

    container.innerHTML = html;

    container.querySelectorAll('.video-thumbnail.youtube-remote').forEach(card => {
        const url = card.dataset.url;
        const videoId = card.dataset.videoId;
        
        // Ensure we have either a videoId or a url
        if (!videoId && !url) {
            logger.warn('YouTube video card missing both videoId and url:', card);
            return;
        }
        
        card.addEventListener('click', (event) => {
            // Don't open player if clicking on action buttons
            if (event.target.closest('.thumbnail-action-btn')) {
                return;
            }
            
            const title = card.querySelector('.thumbnail-name') ? card.querySelector('.thumbnail-name').textContent : 'YouTube Video';
            
            // Prioritize videoId from data attribute, fallback to extracting from URL
            const extractedVideoId = videoId && videoId.trim() ? videoId.trim() : null;
            
            openYouTubeEmbeddedPlayer({
                title,
                url: url || (extractedVideoId ? `https://youtu.be/${extractedVideoId}` : null),
                videoId: extractedVideoId
            });
        });
    });
    
    // Add delete button handlers
    container.querySelectorAll('.youtube-delete-btn').forEach(btn => {
        btn.addEventListener('click', async (event) => {
            event.stopPropagation(); // Prevent card click
            const videoId = btn.dataset.videoId;
            const videoTitle = btn.dataset.videoTitle || 'this video';
            const accountId = btn.dataset.accountId || currentYouTubeAccountId || youtubeActiveAccountId || null;
            
            if (!videoId) {
                return;
            }
            
            const confirmed = confirm(`Are you sure you want to delete "${videoTitle}" from YouTube?\n\nThis action cannot be undone.`);
            if (!confirmed) {
                return;
            }
            
            // Disable button during deletion
            btn.disabled = true;
            btn.style.opacity = '0.5';
            
            try {
                const response = await ipcRenderer.invoke('youtube.video.remove', {
                    videoId: videoId,
                    accountId: accountId
                });
                
                if (!response || !response.success) {
                    throw new Error(response && response.error ? response.error : 'Failed to delete video from YouTube.');
                }
                
                // Success - the library will be refreshed automatically via the event handler
                trackEvent('youtube_video_deleted_from_library', {
                    video_id: videoId
                });
            } catch (error) {
                // Re-enable button on error
                btn.disabled = false;
                btn.style.opacity = '1';
                
                const message = error && error.message ? error.message : 'Failed to delete video from YouTube.';
                alert(`Failed to delete video.\n${message}`);
            }
        });
    });
}

function renderYouTubePlaylistCollection(playlists, options = {}) {
    const container = document.getElementById('galleryContent');
    if (!container) {
        return;
    }

    if (!Array.isArray(playlists) || playlists.length === 0) {
        const heading = options.heading || 'Playlists';
        const emptyMessage = searchQuery
            ? 'No playlists match the current search.'
            : 'No playlists are available yet.';
        container.innerHTML = `
            <div class="empty-state">
                <h2>${escapeHtml(heading)}</h2>
                <p>${escapeHtml(emptyMessage)}</p>
            </div>
        `;
        return;
    }

    const heading = options.heading || 'Playlists';
    const account = youtubeAccounts.find(acc => acc.id === accountId) || null;
    const resolvedAccountLabel = options.accountLabel || (account ? getYouTubeAccountLabel(account) : null);
    const accountLabel = resolvedAccountLabel ? `<p class="section-subheading">${escapeHtml(resolvedAccountLabel)}</p>` : '';
    const accountId = options.accountId || currentYouTubeAccountId || youtubeActiveAccountId || null;

    let html = `
        <div class="folder-section">
            <div class="folder-header">${escapeHtml(heading)}</div>
            ${accountLabel}
            <div class="video-grid">
    `;

    const sortedPlaylists = playlists.slice().sort((a, b) => {
        const nameA = (a && a.title ? a.title : '').toLowerCase();
        const nameB = (b && b.title ? b.title : '').toLowerCase();
        return nameA.localeCompare(nameB);
    });

    sortedPlaylists.forEach((playlist) => {
        if (!playlist) {
            return;
        }
        const thumbnailCandidates = playlist.thumbnails || {};
        const thumbnailUrl = (thumbnailCandidates.high && thumbnailCandidates.high.url)
            || (thumbnailCandidates.medium && thumbnailCandidates.medium.url)
            || (thumbnailCandidates.default && thumbnailCandidates.default.url)
            || '';
        const metaParts = [];
        if (typeof playlist.itemCount === 'number') {
            metaParts.push(`${playlist.itemCount} video${playlist.itemCount === 1 ? '' : 's'}`);
        }
        if (playlist.updatedAt) {
            metaParts.push(`Updated ${formatDate(new Date(playlist.updatedAt))}`);
        }
        const playlistKey = buildYouTubePlaylistKey(accountId, playlist.id);
        html += `
            <div class="video-thumbnail youtube-playlist-card" data-path="${escapeHtml(playlistKey)}">
                ${thumbnailUrl
                    ? `<img class="thumbnail-image" src="${escapeHtml(thumbnailUrl)}" alt="">`
                    : '<div class="thumbnail-placeholder">PL</div>'}
                <div class="thumbnail-info">
                    <div class="thumbnail-name">${escapeHtml(playlist.title || 'Untitled Playlist')}</div>
                    <div class="thumbnail-meta">${escapeHtml(metaParts.join(' • '))}</div>
                </div>
            </div>
        `;
    });

    html += `
            </div>
        </div>
    `;

    container.innerHTML = html;

    container.querySelectorAll('.video-thumbnail.youtube-playlist-card').forEach(card => {
        const path = card.dataset.path;
        if (!path) {
            return;
        }
        card.addEventListener('click', () => {
            filterAndRenderGallery(path);
            updateDirectoryTree();
        });
    });
}

function updateYouTubeStats(items = [], options = {}) {
    const stats = document.getElementById('stats');
    if (!stats) {
        return;
    }

    if (options.loading) {
        stats.textContent = 'Loading YouTube library…';
        return;
    }

    if (!Array.isArray(items) || items.length === 0) {
        if (searchQuery) {
            stats.textContent = 'No YouTube items match this search';
        } else if (options.context && options.context.type === 'playlist') {
            stats.textContent = 'No videos in this playlist';
        } else if (options.isCollection) {
            stats.textContent = 'No playlists available';
        } else {
            stats.textContent = 'No YouTube videos available';
        }
        return;
    }

    let accountLabel = options.accountLabel || null;
    if (!accountLabel && options.context && options.context.accountId) {
        const account = youtubeAccounts.find(acc => acc.id === options.context.accountId);
        if (account) {
            accountLabel = getYouTubeAccountLabel(account);
        }
    }

    let label = 'videos';
    if (options.context) {
        const category = options.context.category ? options.context.category.toLowerCase() : '';
        if (options.context.type === 'playlist' && options.playlist) {
            label = `videos in "${options.playlist.title || 'Playlist'}"`;
            const suffix = searchQuery ? ' matching search' : '';
            const accountSuffix = accountLabel ? ` on ${accountLabel}` : '';
            stats.textContent = `${items.length} ${label}${accountSuffix}${suffix}`;
            return;
        }
        if (category === 'shorts') {
            label = 'shorts';
        } else if (category === 'live') {
            label = 'live streams';
        } else if (category === 'podcasts') {
            label = 'podcast videos';
        } else if (category === 'playlists') {
            label = options.isCollection ? 'playlists' : 'videos';
        }
    }

    if (options.isCollection) {
        const suffix = searchQuery ? ' matching search' : '';
        const accountSuffix = accountLabel ? ` on ${accountLabel}` : '';
        stats.textContent = `${items.length} YouTube ${label}${accountSuffix}${suffix}`;
    } else {
        const suffix = searchQuery ? ' matching search' : '';
        const accountSuffix = accountLabel ? ` on ${accountLabel}` : '';
        stats.textContent = `${items.length} YouTube ${label}${accountSuffix}${suffix}`;
    }
}

// Organize videos by folder structure
function organizeByFolders(videosList = videos) {
    const folderMap = new Map();

    videosList.forEach(video => {
        // Use directory name as top level, then subdirectory path
        const topLevel = video.directoryName || 'Root';
        const subPath = video.directory || '';
        const folderPath = subPath ? `${topLevel}${subPath ? ' / ' + subPath : ''}` : topLevel;

        if (!folderMap.has(folderPath)) {
            folderMap.set(folderPath, []);
        }
        folderMap.get(folderPath).push(video);
    });

    return folderMap;
}

// Render gallery
async function renderGallery(videosList = undefined) {
    const container = document.getElementById('galleryContent');
    const listToRender = typeof videosList === 'undefined' ? getFilteredVideos() : videosList;
    const folderMap = organizeByFolders(listToRender);

    if (listToRender.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <h2>No Videos Found</h2>
                <p>${currentSelectedDirectory || searchQuery ? 'No videos match the current filters' : 'Add directories in settings to start monitoring videos'}</p>
            </div>
        `;
        return;
    }

    let html = '';
    const sortedFolders = Array.from(folderMap.entries()).sort((a, b) => {
        if (a[0] === 'Root') return -1;
        if (b[0] === 'Root') return 1;
        return a[0].localeCompare(b[0]);
    });

    for (const [folderPath, folderVideos] of sortedFolders) {
        html += `
            <div class="folder-section">
                <div class="folder-header">${folderPath}</div>
                <div class="video-grid">
        `;

        for (const video of folderVideos) {
            const thumbnailPath = await ipcRenderer.invoke('get-thumbnail', video.path);
            // Format path for file:// URL
            let thumbnailSrc = null;
            if (thumbnailPath) {
                // Convert Windows backslashes to forward slashes
                    const normalizedPath = thumbnailPath.replace(/\\/g, '/');
                // For Windows absolute paths (C:/...), use file:///C:/...
                // For Unix paths (/...), use file:///...
                if (normalizedPath.match(/^[A-Za-z]:/)) {
                    // Windows absolute path
                    thumbnailSrc = `file:///${normalizedPath}`;
                } else if (normalizedPath.startsWith('/')) {
                    // Unix absolute path
                    thumbnailSrc = `file://${normalizedPath}`;
                } else {
                    // Relative path
                    thumbnailSrc = `file:///${normalizedPath}`;
                }
            }

            const youtubeRecord = getYouTubeRecord(video.path);
            const youtubeJob = getYouTubeJobByVideoPath(video.path, currentYouTubeAccountId || youtubeActiveAccountId || null);
            let youtubeStatusHtml = '';

            if (youtubeJob && youtubeJob.status && youtubeJob.status !== 'failed' && youtubeJob.status !== 'completed') {
                const percent = Number.isFinite(youtubeJob.progress) ? Math.round(Math.max(0, Math.min(100, youtubeJob.progress))) : null;
                const label = percent !== null ? `Uploading… ${percent}%` : 'Uploading…';
                youtubeStatusHtml = `<div class="thumbnail-youtube-status uploading">${label}</div>`;
            } else if (youtubeJob && youtubeJob.status === 'failed') {
                youtubeStatusHtml = `<div class="thumbnail-youtube-status failed">Upload failed</div>`;
            } else if (youtubeRecord && youtubeRecord.videoId) {
                const visibility = (youtubeRecord.privacy || 'unlisted').toUpperCase();
                youtubeStatusHtml = `<div class="thumbnail-youtube-status uploaded">${visibility}</div>`;
            } else if (youtubeRecord && youtubeRecord.lastError) {
                youtubeStatusHtml = `<div class="thumbnail-youtube-status failed">Upload error</div>`;
            }

            html += `
                <div class="video-thumbnail ${currentVideo && currentVideo.path === video.path ? 'active' : ''}" 
                     data-path="${video.path.replace(/"/g, '&quot;')}">
                    ${thumbnailSrc ? 
                        `<img src="${thumbnailSrc}" class="thumbnail-image" onerror="this.parentElement.querySelector('.thumbnail-placeholder').style.display='flex'; this.style.display='none';">` : 
                        ''
                    }
                    <div class="thumbnail-placeholder" style="display: ${thumbnailSrc ? 'none' : 'flex'}">
                        🎬
                    </div>
                    <div class="thumbnail-name">${video.name}</div>
                    <div class="thumbnail-info">
                        <div class="thumbnail-meta">${formatFileSize(video.size)}</div>
                        ${youtubeStatusHtml}
                        <div class="thumbnail-actions">
                            <button class="thumbnail-action-btn edit-video-btn" title="Edit video" data-path="${video.path.replace(/"/g, '&quot;')}">✂</button>
                            <button class="thumbnail-action-btn rename-video-btn" title="Rename video" data-path="${video.path.replace(/"/g, '&quot;')}">✏</button>
                            <button class="thumbnail-action-btn upload-video-btn" title="Upload to YouTube" data-path="${video.path.replace(/"/g, '&quot;')}">⬆</button>
                            <button class="thumbnail-action-btn delete-video-btn" title="Delete video" data-path="${video.path.replace(/"/g, '&quot;')}">🗑</button>
                        </div>
                    </div>
                </div>
            `;
        }

        html += `
                </div>
            </div>
        `;
    }

    container.innerHTML = html;

    // Add click handlers
    container.querySelectorAll('.video-thumbnail').forEach(thumb => {
        thumb.addEventListener('click', () => {
            const path = thumb.dataset.path;
            const video = videoMap.get(path);
            if (video) {
                selectVideo(video);
            }
        });
    });

    container.querySelectorAll('.edit-video-btn').forEach(btn => {
        btn.addEventListener('click', (event) => {
            event.stopPropagation();
            const path = btn.dataset.path;
            const video = videoMap.get(path);
            if (video) {
                openVideoEditor(video);
            }
        });
    });

    container.querySelectorAll('.rename-video-btn').forEach(btn => {
        btn.addEventListener('click', (event) => {
            event.stopPropagation();
            const path = btn.dataset.path;
            const video = videoMap.get(path);
            if (video) {
                renameVideo(video);
            }
        });
    });

    container.querySelectorAll('.upload-video-btn').forEach(btn => {
        btn.addEventListener('click', (event) => {
            event.stopPropagation();
            const path = btn.dataset.path;
            const video = videoMap.get(path);
            if (video) {
                const record = getYouTubeRecord(video.path);
                openYouTubeUploadModal(video, record && record.videoId ? 'update' : 'upload');
            }
        });
    });

    container.querySelectorAll('.delete-video-btn').forEach(btn => {
        btn.addEventListener('click', (event) => {
            event.stopPropagation();
            const path = btn.dataset.path;
            const video = videoMap.get(path);
            if (video) {
                deleteVideo(video);
            }
        });
    });
}


function setRenameError(message = '') {
    if (renameErrorLabel) {
        renameErrorLabel.textContent = message;
    }
}

function updateRenamePreview() {
    if (!renamePreviewLabel) {
        return;
    }

    if (!pendingRenameVideo) {
        renamePreviewLabel.textContent = '';
        return;
    }

    const inputValue = renameInput.value;
    const trimmed = inputValue.trim();

    if (!trimmed) {
        renamePreviewLabel.textContent = renameExtension ? `(will become ${renameExtension})` : '(enter a name)';
        return;
    }

    const extensionLower = renameExtension.toLowerCase();
    let finalName = trimmed;
    if (renameExtension && !trimmed.toLowerCase().endsWith(extensionLower)) {
        finalName = `${trimmed}${renameExtension}`;
    }

    renamePreviewLabel.textContent = finalName;
}

function closeRenameModal() {
    if (!renameModal) {
        return;
    }

    renameModal.classList.remove('active');
    pendingRenameVideo = null;
    renameExtension = '';
    setRenameError('');
    if (renameConfirmBtn) {
        renameConfirmBtn.disabled = false;
        renameConfirmBtn.textContent = 'Rename';
    }
    if (renameInput) {
        renameInput.value = '';
    }
    if (renamePreviewLabel) {
        renamePreviewLabel.textContent = '';
    }
    if (renameExtensionLabel) {
        renameExtensionLabel.textContent = '';
        renameExtensionLabel.classList.add('hidden');
    }
    if (renameCurrentNameLabel) {
        renameCurrentNameLabel.textContent = '';
    }
}

function openRenameModal(video) {
    if (!renameModal || !renameInput) {
        return;
    }

    pendingRenameVideo = video;
    renameExtension = video.extension ? `.${video.extension}` : '';
    const extensionLower = renameExtension.toLowerCase();
    let defaultName = video.name;
    if (renameExtension && video.name.toLowerCase().endsWith(extensionLower)) {
        defaultName = video.name.slice(0, -renameExtension.length);
    }

    renameInput.value = defaultName;
    if (renameCurrentNameLabel) {
        renameCurrentNameLabel.textContent = video.name;
    }

    if (renameExtensionLabel) {
        if (renameExtension) {
            renameExtensionLabel.textContent = renameExtension;
            renameExtensionLabel.classList.remove('hidden');
        } else {
            renameExtensionLabel.textContent = '';
            renameExtensionLabel.classList.add('hidden');
        }
    }

    setRenameError('');
    if (renameConfirmBtn) {
        renameConfirmBtn.disabled = false;
        renameConfirmBtn.textContent = 'Rename';
    }

    renameModal.classList.add('active');
    updateRenamePreview();

    setTimeout(() => {
        renameInput.focus();
        renameInput.select();
    }, 0);
}

async function submitRename() {
    if (!pendingRenameVideo || !renameInput) {
        return;
    }

    const trimmed = renameInput.value.trim();

    if (!trimmed) {
        setRenameError('File name cannot be empty.');
        renameInput.focus();
        return;
    }

    if (INVALID_FILENAME_CHARS.test(trimmed)) {
        setRenameError('File name contains invalid characters (\\ / : * ? " < > |).');
        renameInput.focus();
        return;
    }

    const extensionLower = renameExtension.toLowerCase();
    let finalFileName = trimmed;
    if (renameExtension && !trimmed.toLowerCase().endsWith(extensionLower)) {
        finalFileName = `${trimmed}${renameExtension}`;
    }

    if (finalFileName === pendingRenameVideo.name) {
        closeRenameModal();
        return;
    }

    try {
        if (renameConfirmBtn) {
            renameConfirmBtn.disabled = true;
            renameConfirmBtn.textContent = 'Renaming...';
        }

        const result = await ipcRenderer.invoke('rename-video-file', {
            filePath: pendingRenameVideo.path,
            newFileName: finalFileName
        });

        if (!result || !result.success || !result.updatedVideo) {
            const message = result && result.error ? result.error : 'Unknown error';
            setRenameError(message);
            if (renameConfirmBtn) {
                renameConfirmBtn.disabled = false;
                renameConfirmBtn.textContent = 'Rename';
            }
            return;
        }

        const originalPath = pendingRenameVideo.path;
        closeRenameModal();
        applyVideoRename(originalPath, result.updatedVideo);
        trackEvent('video_renamed', {
            ...getVideoAnalyticsProps(result.updatedVideo),
            name_length: Math.min(finalFileName.length, 180)
        });
    } catch (error) {
        logger.error('Error renaming video:', error);
        setRenameError('Failed to rename video. Please try again.');
        if (renameConfirmBtn) {
            renameConfirmBtn.disabled = false;
            renameConfirmBtn.textContent = 'Rename';
        }
    }
}

function renameVideo(video) {
    if (!video) {
        return;
    }
    openRenameModal(video);
}

if (renameInput) {
    renameInput.addEventListener('input', () => {
        setRenameError('');
        updateRenamePreview();
    });

    renameInput.addEventListener('keydown', (event) => {
        if (event.key === 'Enter' && !event.shiftKey && !event.ctrlKey && !event.metaKey) {
            event.preventDefault();
            submitRename();
        }
    });
}

if (renameConfirmBtn) {
    renameConfirmBtn.addEventListener('click', (event) => {
        event.preventDefault();
        submitRename();
    });
}

if (renameCancelBtn) {
    renameCancelBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeRenameModal();
    });
}

if (renameCloseBtn) {
    renameCloseBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeRenameModal();
    });
}

if (renameModal) {
    renameModal.addEventListener('click', (event) => {
        if (event.target === renameModal) {
            closeRenameModal();
        }
    });
}

if (saveTemplateConfirmBtn) {
    saveTemplateConfirmBtn.addEventListener('click', (event) => {
        event.preventDefault();
        submitSaveTemplate();
    });
}

if (saveTemplateCancelBtn) {
    saveTemplateCancelBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeSaveTemplateModal();
    });
}

if (saveTemplateCloseBtn) {
    saveTemplateCloseBtn.addEventListener('click', (event) => {
        event.preventDefault();
        closeSaveTemplateModal();
    });
}

if (saveTemplateModal) {
    saveTemplateModal.addEventListener('click', (event) => {
        if (event.target === saveTemplateModal) {
            closeSaveTemplateModal();
        }
    });
}

async function deleteVideo(video) {
    const confirmed = confirm(`Delete "${video.name}"? This action cannot be undone.`);
    if (!confirmed) {
        return;
    }

    try {
        const result = await ipcRenderer.invoke('delete-video-file', video.path);
        if (!result || !result.success) {
            const message = result && result.error ? result.error : 'Unknown error';
            alert(`Failed to delete video.\n${message}`);
            return;
        }

        removeVideoFromState(video.path);
        trackEvent('video_deleted', {
            ...getVideoAnalyticsProps(video),
            trigger: 'user_action'
        });
    } catch (error) {
        logger.error('Error deleting video:', error);
        alert('Failed to delete video. Please try again.');
    }
}

// Close video and return to gallery view
function closeVideo() {
    const previousVideo = currentVideo;
    const previousViewMode = viewMode;
    const playerContainer = document.getElementById('playerContainer');
    const videoElement = playerContainer ? playerContainer.querySelector('video') : null;

    if (videoElement) {
        cleanupVideoElement(videoElement);
    }

    if (playerContainer) {
        playerContainer.innerHTML = '';
        playerContainer.style.display = 'none';
    }

    currentVideo = null;
    state.currentVideo = currentVideo;
    viewMode = 'gallery';
    state.viewMode = viewMode;
    editorState = null;
    state.editorState = editorState;

    // Hide details
    document.getElementById('videoDetails').style.display = 'none';

    // Show gallery
    document.getElementById('galleryContainer').style.display = 'block';

    // Show sidebar if there are monitored directories
    if (settings.monitoredDirectories && settings.monitoredDirectories.length > 0) {
        document.getElementById('folderSidebar').style.display = 'block';
    }

    // Render gallery to update active state, preserving directory filter if one is selected
    filterAndRenderGallery(currentTreeSelection);

    trackScreenView('gallery', { trigger: 'close_video', previous_view: previousViewMode });
    if (previousVideo) {
        trackEvent('viewer_video_closed', {
            ...getVideoAnalyticsProps(previousVideo),
            previous_view: previousViewMode
        });
    }
}

// Select video
async function selectVideo(video) {
    const previousViewMode = viewMode;
    currentVideo = video;
    state.currentVideo = currentVideo;
    viewMode = 'player';
    state.viewMode = viewMode;
    editorState = null;
    state.editorState = editorState;

    // Update UI
    document.getElementById('galleryContainer').style.display = 'none';
    document.getElementById('playerContainer').style.display = 'flex';
    const sidebar = document.getElementById('folderSidebar');
    sidebar.style.display = 'none';
    sidebar.classList.remove('open');

    // Render gallery to update active state
    renderGallery();

    // Show video player with close button
    const container = document.getElementById('playerContainer');
    cleanupPlaybackContainer();
    container.innerHTML = `
        <button class="close-video-btn" id="closeVideoBtn" title="Close video">×</button>
        <div style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; position: relative;">
            <div class="player-status-banner" style="position:absolute; top:16px; right:16px; background:rgba(0,0,0,0.65); padding:6px 12px; border-radius:4px; font-size:13px; color:#fff; display:none;"></div>
            <div class="player-error-banner" style="position:absolute; left:50%; transform:translateX(-50%); bottom:32px; background:rgba(128,0,0,0.8); padding:10px 16px; border-radius:4px; font-size:14px; color:#fff; display:none; max-width:80%; text-align:center;"></div>
            <video class="video-player" controls style="max-width: 100%; max-height: 100%;"></video>
        </div>
    `;

    // Add event listener for close button
    document.getElementById('closeVideoBtn').addEventListener('click', closeVideo);

    const videoElement = container.querySelector('video');
    const statusElement = container.querySelector('.player-status-banner');
    const errorElement = container.querySelector('.player-error-banner');

    try {
        await setupVideoPlayback(videoElement, video, { statusElement, errorElement });
    } catch (error) {
        logger.error('Failed to initialize video playback:', error);
        if (videoElement) {
            videoElement.style.display = 'none';
        }
    }

    // Show details
    const details = document.getElementById('videoDetails');
    details.style.display = 'block';
    document.getElementById('detailsTitle').textContent = video.name;

    document.getElementById('detailsInfo').innerHTML = `
        <div class="detail-item">
            <strong>File Path</strong>
            ${video.path}
        </div>
        <div class="detail-item">
            <strong>Directory</strong>
            ${video.directory || 'Root'}
        </div>
        <div class="detail-item">
            <strong>File Size</strong>
            ${formatFileSize(video.size)}
        </div>
        <div class="detail-item">
            <strong>Created</strong>
            ${formatDate(video.created)}
        </div>
        <div class="detail-item">
            <strong>Modified</strong>
            ${formatDate(video.modified)}
        </div>
        <div class="detail-item">
            <strong>YouTube</strong>
            <div class="youtube-details-content" id="youtubeDetailsContent"></div>
        </div>
    `;
    renderYouTubeDetail(video);

    trackScreenView('player', { trigger: 'open_video', previous_view: previousViewMode });
    trackEvent('viewer_video_opened', {
        ...getVideoAnalyticsProps(video),
        previous_view: previousViewMode
    });
}

async function openVideoEditor(video) {
    if (!video) {
        return;
    }

    const previousViewMode = viewMode;
    currentVideo = video;
    state.currentVideo = currentVideo;
    viewMode = 'editor';
    state.viewMode = viewMode;

    const galleryContainer = document.getElementById('galleryContainer');
    const sidebar = document.getElementById('folderSidebar');
    const playerContainer = document.getElementById('playerContainer');
    const details = document.getElementById('videoDetails');

    if (galleryContainer) {
        galleryContainer.style.display = 'none';
    }
    if (sidebar) {
        sidebar.style.display = 'none';
        sidebar.classList.remove('open');
    }
    if (details) {
        details.style.display = 'none';
    }

    if (!playerContainer) {
        return;
    }

    cleanupPlaybackContainer();

    const extensionLabel = video.extension ? video.extension.toUpperCase() : '';
    const infoParts = [formatFileSize(video.size)];
    if (extensionLabel) {
        infoParts.push(extensionLabel);
    }
    const infoLine = infoParts.join(' • ');

    playerContainer.style.display = 'flex';
    playerContainer.innerHTML = `
        <button class="close-video-btn" id="closeVideoBtn" title="Back to gallery">×</button>
        <div class="editor-layout">
            <div class="editor-heading">
                <h2>${video.name}</h2>
                <div class="editor-heading-meta">${infoLine}</div>
            </div>
            <div class="editor-video-wrapper">
                <div class="player-status-banner" style="position:absolute; top:16px; right:16px; background:rgba(0,0,0,0.65); padding:6px 12px; border-radius:4px; font-size:13px; color:#fff; display:none;"></div>
                <div class="player-error-banner" style="position:absolute; left:50%; transform:translateX(-50%); bottom:32px; background:rgba(128,0,0,0.8); padding:10px 16px; border-radius:4px; font-size:14px; color:#fff; display:none; max-width:80%; text-align:center;"></div>
                <video class="video-player" id="editorVideo" controls></video>
            </div>
            <div class="editor-timeline">
                <div class="timeline-header">
                    <div class="timeline-title">
                        <span class="timeline-label">Timeline</span>
                        <span class="timeline-summary" id="editorSegmentSummary">No splits</span>
                    </div>
                    <div class="timeline-actions">
                        <button class="btn-icon" id="videoInfoBtn" title="Video file info">Info</button>
                        <button class="btn-icon" id="splitSegmentBtn" title="Split at playhead (S)" disabled>
                            <svg viewBox="0 0 24 24" aria-hidden="true" focusable="false">
                                <path d="M9.64 7.64 8 9.27l-.82-.82a2.5 2.5 0 1 0-3.54 3.54l5.47 5.47a2.5 2.5 0 1 0 3.54-3.54l-.82-.82 1.63-1.63L19 19.25V21h-1.75L9.64 12.39l-.82.82a2.5 2.5 0 0 1-3.54-3.54l.82-.82L4.81 7.1 6.22 5.7l1.59 1.59 1.83-1.83L9.64 7.64zm4.61-4.61 1.41-1.41 4.24 4.24-1.41 1.41z" />
                            </svg>
                            <span>Split (S)</span>
                        </button>
                        <button class="btn-icon" id="resetSplitsBtn" disabled>Reset</button>
                    </div>
                </div>
                <div class="timeline-scale" id="editorTimelineScale"></div>
                <div class="timeline-track" id="editorTimelineTrack">
                    <div class="timeline-segments" id="editorTimelineSegments"></div>
                    <div class="timeline-cue-points" id="editorTimelineCuePoints"></div>
                    <div class="timeline-playhead" id="editorTimelinePlayhead"></div>
                </div>
                <input type="range" id="editorTimeline" min="0" max="0" value="0" step="0.01">
                <div class="editor-time-display">
                    <span id="editorCurrentTime">00:00.00</span>
                    <span>/</span>
                    <span id="editorTotalTime">00:00.00</span>
                </div>
                <div class="editor-highlight-detect">
                    <div class="highlight-detect-header">
                        <span class="highlight-detect-title">Detect highlights</span>
                    </div>
                    <div class="detection-mode-tabs">
                        <button class="tab active" id="audioModeTab" data-mode="audio">Audio</button>
                        <button class="tab" id="visualModeTab" data-mode="visual">Visual</button>
                        
                    </div>

                    <div class="template-controls-row">
                        <button class="btn-icon btn-small" id="highlightSaveTemplateBtn" title="Save current sample as template" disabled>
                            Save Template
                        </button>
                        <label class="highlight-window-label">
                            Load:
                            <select id="highlightTemplateSelect" class="highlight-template-select" title="Load a saved template">
                                <option value="">— Select template —</option>
                            </select>
                        </label>
                        <button class="btn-icon btn-small" id="highlightDeleteTemplateBtn" title="Delete selected template" style="display: none;">
                            Delete
                        </button>
                        <button class="btn-icon btn-small" id="highlightExportTemplateBtn" title="Export selected template" style="display: none;">
                            Export
                        </button>
                        <button class="btn-icon btn-small" id="highlightImportTemplateBtn" title="Import a template">
                            Import
                        </button>
                    </div>
                    
                    <div class="detection-panel active" id="audioPanel">
                        <div class="highlight-waveform-container">
                            <canvas id="highlightWaveformCanvas" class="highlight-waveform-canvas" width="800" height="80"></canvas>
                            <div class="highlight-waveform-range" id="highlightWaveformRange"></div>
                            <div class="highlight-waveform-marker" id="highlightMarker"></div>
                            <div class="highlight-waveform-loading" id="highlightWaveformLoading">Loading waveform...</div>
                            <div class="highlight-waveform-overlay" id="highlightWaveformOverlay"></div>
                        </div>
                        <div class="highlight-detect-controls">
                            <span class="highlight-detect-range" id="highlightDetectRange">Click waveform to set marker</span>
                            <label class="highlight-window-label">
                                Window:
                                <input type="number" id="highlightWindowMs" class="highlight-window-input" value="500" min="100" max="5000" step="100"> ms
                            </label>
                            <label class="highlight-window-label">
                                Threshold:
                                <input type="number" id="highlightThreshold" class="highlight-window-input" value="0.3" min="0.3" max="0.95" step="0.05">
                            </label>
                            <label class="highlight-window-label">
                                Min gap:
                                <input type="number" id="highlightMinGap" class="highlight-window-input" value="1" min="0.5" max="10" step="0.5"> s
                            </label>
                            <button class="btn-icon btn-small" id="highlightPlayLoopBtn" title="Play selected clip on loop" disabled>
                                Play loop
                            </button>
                            <button class="btn-icon btn-small" id="highlightScanBtn" title="Scan video for this sound" disabled>
                                Scan
                            </button>
                            <button class="btn-icon btn-small" id="highlightClearBtn" title="Clear detected markers" disabled>
                                Clear
                            </button>
                        </div>
                    </div>
                    
                    <div class="detection-panel" id="visualPanel">
                        <div class="visual-detection-header">
                            <span style="font-size: 12px; color: #888;">Click "Select Region" then drag on video to select template area</span>
                        </div>
                        <div class="video-region-actions">
                            <button class="btn-icon btn-small" id="selectRegionBtn" title="Click then drag on video to select a region">
                                Select Region
                            </button>
                            <button class="btn-icon btn-small" id="visualScanBtn" title="Select a region first" disabled>
                                Scan
                            </button>
                            <button class="btn-icon btn-small" id="viewVisualResultsBtn" title="Perform a scan to see results" disabled>
                                View Results
                            </button>
                        </div>
                        <div class="visual-method-select">
                            <select id="visualMethodSelect" name="visualMethod">
                                <option value="gpu" selected>GPU (MobileNet AI)</option>
                                <option value="fast">Fast (Resize + MSE)</option>
                                <option value="balanced">Balanced (pixelmatch)</option>
                                <option value="accurate">Accurate (Direct)</option>
                                <option value="phash">pHash (Perceptual)</option>
                                <option value="ssim">SSIM (Structural)</option>
                                <option value="multiscale">Multi-Scale + Histogram</option>
                            </select>
                        </div>
                        <div class="visual-threshold-input">
                            <span>Similarity threshold:</span>
                            <input type="number" id="visualThreshold" value="0.7" min="0.3" max="0.99" step="0.05">
                        </div>
                        <div class="visual-frame-interval">
                            <span>Frame skip:</span>
                            <input type="number" id="visualFrameSkip" value="5" min="1" max="30" step="1">
                        </div>
                        <div class="visual-frame-interval">
                            <span>Batch size:</span>
                            <input type="number" id="visualBatchSize" value="128" min="16" max="256" step="16">
                        </div>
                        <div class="visual-frame-interval">
                            <span>Min gap:</span>
                            <input type="number" id="visualMinGap" value="1" min="0.5" max="300" step="0.5"> s
                        </div>
                        <div class="visual-template-preview" id="visualTemplatePreview" style="display: none;">
                            <img id="visualTemplateImage" src="" alt="Template preview">
                            <div class="visual-template-info">
                                <div class="template-name" id="visualTemplateName">Template ready</div>
                                <div>Method: <span id="visualTemplateMethod">-</span></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="highlight-detect-progress" id="highlightDetectProgress" style="display:none;">
                        <div class="highlight-detect-progress-bar"><div class="highlight-detect-progress-fill" id="highlightDetectProgressFill"></div></div>
                        <span class="highlight-detect-progress-text" id="highlightDetectProgressText"></span>
                    </div>
                </div>
            </div>
            <div class="editor-actions-row">
                <div class="editor-status info" id="editorStatus"></div>
                <div class="editor-action-buttons">
                    <button class="btn btn-secondary" id="cancelEditorBtn">Cancel</button>
                    <button class="btn" id="exportEditorBtn" disabled>Export Edited Video</button>
                </div>
            </div>
        </div>
    `;

    renderGallery();
    trackScreenView('editor', { trigger: 'open_editor', previous_view: previousViewMode });
    trackEvent('editor_opened', {
        ...getVideoAnalyticsProps(video),
        previous_view: previousViewMode
    });

    const videoElement = document.getElementById('editorVideo');
    const timelineElement = document.getElementById('editorTimeline');
    const videoInfoBtn = document.getElementById('videoInfoBtn');
    const splitBtn = document.getElementById('splitSegmentBtn');
    const resetSplitsBtn = document.getElementById('resetSplitsBtn');
    const exportBtn = document.getElementById('exportEditorBtn');
    const statusElement = document.getElementById('editorStatus');
    const currentTimeLabel = document.getElementById('editorCurrentTime');
    const totalTimeLabel = document.getElementById('editorTotalTime');
    const timelineTrack = document.getElementById('editorTimelineTrack');
    const timelineSegmentsElement = document.getElementById('editorTimelineSegments');
    const timelinePlayhead = document.getElementById('editorTimelinePlayhead');
    const timelineScale = document.getElementById('editorTimelineScale');
    const segmentSummaryElement = document.getElementById('editorSegmentSummary');
    const playbackStatusElement = playerContainer.querySelector('.player-status-banner');
    const playbackErrorElement = playerContainer.querySelector('.player-error-banner');
    const highlightWindowMs = document.getElementById('highlightWindowMs');
    const highlightThreshold = document.getElementById('highlightThreshold');
    const highlightMinGap = document.getElementById('highlightMinGap');
    const highlightPlayLoopBtn = document.getElementById('highlightPlayLoopBtn');
    const highlightScanBtn = document.getElementById('highlightScanBtn');
    const highlightSaveTemplateBtn = document.getElementById('highlightSaveTemplateBtn');
    const highlightTemplateSelect = document.getElementById('highlightTemplateSelect');
    const highlightDeleteTemplateBtn = document.getElementById('highlightDeleteTemplateBtn');
    const highlightExportTemplateBtn = document.getElementById('highlightExportTemplateBtn');
    const highlightImportTemplateBtn = document.getElementById('highlightImportTemplateBtn');
    const highlightClearBtn = document.getElementById('highlightClearBtn');
    const highlightDetectRange = document.getElementById('highlightDetectRange');
    const highlightWaveformCanvas = document.getElementById('highlightWaveformCanvas');
    const highlightWaveformLoading = document.getElementById('highlightWaveformLoading');
    const highlightWaveformOverlay = document.getElementById('highlightWaveformOverlay');
    const highlightMarker = document.getElementById('highlightMarker');
    const highlightWaveformRange = document.getElementById('highlightWaveformRange');
    const highlightDetectProgress = document.getElementById('highlightDetectProgress');
    const highlightDetectProgressFill = document.getElementById('highlightDetectProgressFill');
    const highlightDetectProgressText = document.getElementById('highlightDetectProgressText');

    editorState = {
        video,
        duration: 0,
        boundaries: [0],
        segmentStates: new Map(),
        segments: [],
        cuePoints: [],
        templateMarker: null,
        templateWindowMs: 500,
        templateThreshold: 0.3,
        templateMinGap: 1,
        waveformPeaks: null,
        templateLoopPlaying: false,
        isScrubbing: false,
        exporting: false,
        highlightScanning: false,
        currentTemplate: null,
        detectionMode: 'audio',
        visualRegion: null,
        visualTemplateImage: null,
        visualMethod: 'gpu',
        visualThreshold: 0.7,
        visualFrameSkip: 5,
        visualBatchSize: 128,
        visualMinGap: 1,
        lastVisualScanResults: [],
        visualRegionSelecting: false,
        elements: {
            videoElement,
            timelineElement,
            splitBtn,
            resetSplitsBtn,
            exportBtn,
            statusElement,
            currentTimeLabel,
            totalTimeLabel,
            timelineTrack,
            timelineSegmentsElement,
            timelinePlayhead,
            timelineScale,
            segmentSummaryElement,
            highlightWindowMs,
            highlightThreshold,
            highlightMinGap,
            highlightPlayLoopBtn,
            highlightScanBtn,
            highlightSaveTemplateBtn,
            highlightTemplateSelect,
            highlightDeleteTemplateBtn,
            highlightExportTemplateBtn,
            highlightImportTemplateBtn,
            highlightClearBtn,
            highlightDetectRange,
            highlightWaveformCanvas,
            highlightWaveformLoading,
            highlightWaveformOverlay,
            highlightMarker,
            highlightWaveformRange,
            highlightDetectProgress,
            highlightDetectProgressFill,
            highlightDetectProgressText
        }
    };
    state.editorState = editorState;

    const closeBtn = document.getElementById('closeVideoBtn');
    if (closeBtn) {
        closeBtn.addEventListener('click', closeVideo);
    }

    const cancelEditorBtn = document.getElementById('cancelEditorBtn');
    if (cancelEditorBtn) {
        cancelEditorBtn.addEventListener('click', closeVideo);
    }

    if (videoElement) {
        videoElement.addEventListener('loadedmetadata', handleEditorLoadedMetadata);
        videoElement.addEventListener('timeupdate', handleEditorTimeUpdate);

        const editorContainer = document.getElementById('playerContainer');
        
        let selectionOverlay = editorContainer.querySelector('.video-overlay-for-selection');
        if (!selectionOverlay) {
            selectionOverlay = document.createElement('div');
            selectionOverlay.className = 'video-overlay-for-selection';
            editorContainer.appendChild(selectionOverlay);
        }

        let regionStartX = 0;
        let regionStartY = 0;
        let isSelectingRegion = false;
        let regionSelectionBox = null;

        function getVideoPosition(e) {
            if (!editorContainer || !videoElement) return null;
            const videoRect = videoElement.getBoundingClientRect();
            const x = e.clientX - videoRect.left;
            const y = e.clientY - videoRect.top;
            return {
                relX: Math.max(0, Math.min(x, videoRect.width)),
                relY: Math.max(0, Math.min(y, videoRect.height)),
                width: videoRect.width,
                height: videoRect.height
            };
        }

        function createRegionBox() {
            if (regionSelectionBox) {
                regionSelectionBox.remove();
            }
            regionSelectionBox = document.createElement('div');
            regionSelectionBox.className = 'video-region-selector active';
            regionSelectionBox.innerHTML = '<div class="selection-box"></div><div class="selection-info"></div>';
            editorContainer.appendChild(regionSelectionBox);
            return regionSelectionBox;
        }

        function removeRegionBox() {
            if (regionSelectionBox) {
                regionSelectionBox.remove();
                regionSelectionBox = null;
            }
        }

        function showRegionBox(relX, relY, width, height) {
            if (!regionSelectionBox) return;
            const box = regionSelectionBox.querySelector('.selection-box');
            const info = regionSelectionBox.querySelector('.selection-info');
            
            const videoRect = videoElement.getBoundingClientRect();
            const containerRect = editorContainer.getBoundingClientRect();
            
            const videoLeft = videoRect.left - containerRect.left;
            const videoTop = videoRect.top - containerRect.top;
            
            if (box) {
                box.style.left = `${videoLeft + relX}px`;
                box.style.top = `${videoTop + relY}px`;
                box.style.width = `${width}px`;
                box.style.height = `${height}px`;
            }
            if (info) {
                info.textContent = `${Math.round(width)}x${Math.round(height)}`;
            }
        }

        async function extractRegionAtPosition(relX, relY, width, height) {
            if (width < 10 || height < 10) {
                setEditorStatus('Region too small. Select a larger area.', 'error');
                return;
            }

            const videoWidth = videoElement.videoWidth || videoElement.videoHeight ? (videoElement.videoWidth * videoElement.videoHeight / videoElement.videoHeight) : videoElement.clientWidth;
            const videoHeight = videoElement.videoHeight || videoElement.videoWidth ? videoElement.videoHeight : videoElement.clientHeight;
            
            editorState.visualRegion = {
                x: Math.round(relX),
                y: Math.round(relY),
                width: Math.round(width),
                height: Math.round(height),
                displayWidth: videoElement.clientWidth,
                displayHeight: videoElement.clientHeight,
                videoWidth: videoElement.videoWidth || videoElement.clientWidth,
                videoHeight: videoElement.videoHeight || videoElement.clientHeight
            };

            const currentTime = videoElement.currentTime;
            console.log('[RegionSelect] Region:', editorState.visualRegion, 'Video dims:', videoElement.videoWidth, 'x', videoElement.videoHeight, 'Display dims:', videoElement.clientWidth, 'x', videoElement.clientHeight);
            setEditorStatus('Extracting visual template...', 'info');

            try {
                const result = await ipcRenderer.invoke('extract-visual-region', {
                    videoPath: editorState.video.path,
                    timestamp: currentTime,
                    region: editorState.visualRegion
                });

                if (result && result.success && result.template) {
                    editorState.visualTemplateImage = result.template.imageData;
                    
                    const visualTemplatePreview = document.getElementById('visualTemplatePreview');
                    const visualTemplateImage = document.getElementById('visualTemplateImage');
                    const visualTemplateMethod = document.getElementById('visualTemplateMethod');
                    const visualScanBtn = document.getElementById('visualScanBtn');
                    const selectRegionBtn = document.getElementById('selectRegionBtn');
                    
                    if (visualTemplatePreview) visualTemplatePreview.style.display = 'flex';
                    if (visualTemplateImage) {
                        visualTemplateImage.src = `data:image/png;base64,${result.template.imageData}`;
                    }
                    if (visualTemplateMethod) {
                        const methodNames = { fast: 'Fast (Resize+MSE)', balanced: 'Balanced (pixelmatch)', accurate: 'Accurate (OpenCV)', gpu: 'GPU (MobileNet AI)' };
                        visualTemplateMethod.textContent = methodNames[editorState.visualMethod] || 'Fast';
                    }
                    if (visualScanBtn) { visualScanBtn.disabled = false; visualScanBtn.title = 'Scan video for visual matches'; }
                    if (selectRegionBtn) selectRegionBtn.textContent = 'Change Region';
                    
                    setEditorStatus(`Template extracted (${Math.round(width)}x${Math.round(height)}). Ready to scan.`, 'success');
                } else {
                    setEditorStatus(result?.error || 'Failed to extract visual template.', 'error');
                }
            } catch (err) {
                logger.warn('[Editor] Visual region extraction error:', err);
                setEditorStatus(err?.message || 'Visual region extraction failed.', 'error');
            }
        }

        selectionOverlay.addEventListener('mousedown', (e) => {
            e.preventDefault();
            e.stopPropagation();
            if (!editorState.visualRegionSelecting) return;
            const pos = getVideoPosition(e);
            if (!pos) return;
            isSelectingRegion = true;
            regionStartX = pos.relX;
            regionStartY = pos.relY;
            createRegionBox();
        });

        selectionOverlay.addEventListener('mousemove', (e) => {
            e.preventDefault();
            e.stopPropagation();
            if (!isSelectingRegion || !editorState.visualRegionSelecting) return;
            const pos = getVideoPosition(e);
            if (!pos) return;
            
            const x = Math.min(regionStartX, pos.relX);
            const y = Math.min(regionStartY, pos.relY);
            const width = Math.abs(pos.relX - regionStartX);
            const height = Math.abs(pos.relY - regionStartY);
            
            showRegionBox(x, y, width, height);
        });

        selectionOverlay.addEventListener('mouseup', async (e) => {
            if (!isSelectingRegion || !editorState.visualRegionSelecting) return;
            const pos = getVideoPosition(e);
            if (!pos) return;
            
            const x = Math.min(regionStartX, pos.relX);
            const y = Math.min(regionStartY, pos.relY);
            const width = Math.abs(pos.relX - regionStartX);
            const height = Math.abs(pos.relY - regionStartY);
            
            isSelectingRegion = false;
            editorState.visualRegionSelecting = false;
            selectionOverlay.classList.remove('active');
            removeRegionBox();
            
            await extractRegionAtPosition(x, y, width, height);
        });

        selectionOverlay.addEventListener('mouseleave', () => {
            if (isSelectingRegion) {
                isSelectingRegion = false;
                editorState.visualRegionSelecting = false;
                selectionOverlay.classList.remove('active');
                removeRegionBox();
                const selectRegionBtn = document.getElementById('selectRegionBtn');
                if (selectRegionBtn) {
                    selectRegionBtn.textContent = 'Select Region';
                }
            }
        });
    }

    if (timelineElement) {
        timelineElement.addEventListener('input', handleEditorScrubInput);
        timelineElement.addEventListener('change', handleEditorScrubEnd);
    }

    if (videoInfoBtn) {
        videoInfoBtn.addEventListener('click', showVideoInfoModal);
    }

    if (splitBtn) {
        splitBtn.addEventListener('click', addEditorSplit);
    }

    if (resetSplitsBtn) {
        resetSplitsBtn.addEventListener('click', resetEditorSplits);
    }

    if (timelineTrack) {
        timelineTrack.addEventListener('click', (event) => {
            const rect = timelineTrack.getBoundingClientRect();
            if (rect.width === 0 || !Number.isFinite(editorState.duration) || editorState.duration <= 0) {
                return;
            }

            const clickRatio = Math.min(Math.max((event.clientX - rect.left) / rect.width, 0), 1);
            const newTime = clickRatio * editorState.duration;
            editorState.isScrubbing = false;
            seekEditorTo(newTime);
        });
    }

    if (exportBtn) {
        exportBtn.addEventListener('click', exportEditedVideo);
    }

    if (highlightWindowMs) {
        highlightWindowMs.addEventListener('change', () => {
            if (editorState) {
                const val = parseInt(highlightWindowMs.value, 10);
                editorState.templateWindowMs = isNaN(val) ? 1500 : Math.max(100, Math.min(5000, val));
                updateHighlightDetectUI();
            }
        });
    }
    if (highlightThreshold) {
        highlightThreshold.addEventListener('change', () => {
            if (editorState) {
                const val = parseFloat(highlightThreshold.value);
                editorState.templateThreshold = isNaN(val) ? 0.75 : Math.max(0.3, Math.min(0.95, val));
            }
        });
    }
    if (highlightMinGap) {
        highlightMinGap.addEventListener('change', () => {
            if (editorState) {
                const val = parseFloat(highlightMinGap.value);
                editorState.templateMinGap = isNaN(val) ? 2 : Math.max(0.5, Math.min(10, val));
            }
        });
    }
    if (highlightScanBtn) {
        highlightScanBtn.addEventListener('click', runHighlightDetection);
    }
    if (highlightSaveTemplateBtn) {
        highlightSaveTemplateBtn.addEventListener('click', openSaveTemplateModal);
    }
    if (highlightTemplateSelect) {
        highlightTemplateSelect.addEventListener('change', onHighlightTemplateSelectChange);
    }
    if (highlightDeleteTemplateBtn) {
        highlightDeleteTemplateBtn.addEventListener('click', deleteSelectedTemplate);
    }
    if (highlightExportTemplateBtn) {
        highlightExportTemplateBtn.addEventListener('click', exportSelectedTemplate);
    }
    if (highlightImportTemplateBtn) {
        highlightImportTemplateBtn.addEventListener('click', importTemplate);
    }
    if (highlightClearBtn) {
        highlightClearBtn.addEventListener('click', clearAudioHighlightMarkers);
    }

    const audioModeTab = document.getElementById('audioModeTab');
    const visualModeTab = document.getElementById('visualModeTab');
    const audioPanel = document.getElementById('audioPanel');
    const visualPanel = document.getElementById('visualPanel');

    function switchDetectionMode(mode) {
        editorState.detectionMode = mode;
        if (audioModeTab) audioModeTab.classList.toggle('active', mode === 'audio');
        if (visualModeTab) visualModeTab.classList.toggle('active', mode === 'visual');
        if (audioPanel) audioPanel.classList.toggle('active', mode === 'audio');
        if (visualPanel) visualPanel.classList.toggle('active', mode === 'visual');
        populateHighlightTemplates(mode);
    }

    if (audioModeTab) audioModeTab.addEventListener('click', () => switchDetectionMode('audio'));
    if (visualModeTab) visualModeTab.addEventListener('click', () => switchDetectionMode('visual'));

    const selectRegionBtn = document.getElementById('selectRegionBtn');
    const visualScanBtn = document.getElementById('visualScanBtn');
    const visualThresholdInput = document.getElementById('visualThreshold');
    const visualFrameSkipInput = document.getElementById('visualFrameSkip');
    const visualBatchSizeInput = document.getElementById('visualBatchSize');
    const visualTemplatePreview = document.getElementById('visualTemplatePreview');
    const visualTemplateImage = document.getElementById('visualTemplateImage');
    const visualTemplateMethod = document.getElementById('visualTemplateMethod');

    if (selectRegionBtn) {
        selectRegionBtn.addEventListener('click', () => {
            const playerContainer = document.getElementById('playerContainer');
            const overlay = playerContainer ? playerContainer.querySelector('.video-overlay-for-selection') : null;
            if (overlay) {
                overlay.classList.add('active');
            }
            editorState.visualRegionSelecting = true;
            setEditorStatus('Click and drag on the video to select a region', 'info');
            selectRegionBtn.textContent = 'Selecting...';
        });
    }

    if (visualScanBtn) {
        visualScanBtn.addEventListener('click', runVisualDetection);
    }

    const viewVisualResultsBtn = document.getElementById('viewVisualResultsBtn');
    if (viewVisualResultsBtn) {
        viewVisualResultsBtn.addEventListener('click', () => {
            showVisualResultsModal();
        });
    }


    if (visualThresholdInput) {
        visualThresholdInput.addEventListener('change', () => {
            const val = parseFloat(visualThresholdInput.value);
            editorState.visualThreshold = isNaN(val) ? 0.7 : Math.max(0.3, Math.min(0.99, val));
        });
    }

    if (visualFrameSkipInput) {
        visualFrameSkipInput.addEventListener('change', () => {
            const val = parseInt(visualFrameSkipInput.value);
            editorState.visualFrameSkip = isNaN(val) ? 5 : Math.max(1, Math.min(30, val));
        });
    }

    if (visualBatchSizeInput) {
        visualBatchSizeInput.addEventListener('change', () => {
            const val = parseInt(visualBatchSizeInput.value);
            editorState.visualBatchSize = isNaN(val) ? 128 : Math.max(16, Math.min(256, val));
        });
    }

    const visualMinGapInput = document.getElementById('visualMinGap');
    if (visualMinGapInput) {
        visualMinGapInput.addEventListener('change', () => {
            const val = parseFloat(visualMinGapInput.value);
            editorState.visualMinGap = isNaN(val) ? 1 : Math.max(0.5, val);
        });
    }

    const visualMethodSelect = document.getElementById('visualMethodSelect');
    if (visualMethodSelect) {
        visualMethodSelect.addEventListener('change', (e) => {
            editorState.visualMethod = e.target.value;
        });
    }

    populateHighlightTemplates();
    if (highlightPlayLoopBtn) {
        highlightPlayLoopBtn.addEventListener('click', toggleTemplateLoopPlayback);
    }
    setupHighlightWaveformInteraction();

    try {
        await setupVideoPlayback(videoElement, video, { statusElement: playbackStatusElement, errorElement: playbackErrorElement });
    } catch (error) {
        logger.error('Failed to initialize editor playback:', error);
        if (videoElement) {
            videoElement.style.display = 'none';
        }
        setEditorStatus('Failed to prepare video playback. Ensure the live transcoder is running.', 'error');
        return;
    }

    setEditorStatus('Loading video metadata...', 'info');
}

function handleEditorLoadedMetadata() {
    if (!editorState || !editorState.elements || !editorState.elements.videoElement) {
        return;
    }

    const { videoElement, timelineElement, totalTimeLabel } = editorState.elements;
    editorState.duration = Number(videoElement.duration) || 0;

    if (timelineElement) {
        timelineElement.max = editorState.duration.toFixed(3);
        timelineElement.value = '0';
    }

    if (totalTimeLabel) {
        totalTimeLabel.textContent = formatTimestamp(editorState.duration, 2);
    }

    editorState.boundaries = [0, Number(editorState.duration.toFixed(3))];
    editorState.segmentStates = new Map();

    renderEditorTimelineScale();
    updateTimelinePlayhead(0);

    updateEditorControlsAvailability();
    updateEditorSegments();
    updateHighlightDetectUI();

    loadHighlightWaveform();
    
    // Load cue points from video file
    loadEditorCuePoints();
    
    setEditorStatus('Video ready. Add split points and choose segments to export.', 'info');
}

async function loadEditorCuePoints() {
    logger.log('[Editor] loadEditorCuePoints called');
    if (!editorState || !editorState.video || !editorState.video.path) {
        logger.warn('[Editor] Cannot load cue points: editorState or video path missing');
        logger.warn('[Editor] editorState:', !!editorState);
        logger.warn('[Editor] editorState.video:', editorState ? !!editorState.video : false);
        logger.warn('[Editor] editorState.video.path:', editorState && editorState.video ? editorState.video.path : 'missing');
        return;
    }

    logger.log('[Editor] Loading cue points for:', editorState.video.path);
    try {
        const result = await ipcRenderer.invoke('read-cue-points', editorState.video.path);
        logger.log('[Editor] IPC result:', result);
        logger.log('[Editor] Result success:', result ? result.success : 'no result');
        logger.log('[Editor] Result cuePoints:', result && result.cuePoints ? result.cuePoints.length : 'none');
        
        if (result && result.success && Array.isArray(result.cuePoints)) {
            editorState.cuePoints = result.cuePoints;
            logger.log('[Editor] Stored', result.cuePoints.length, 'cue points in editorState');
            logger.log('[Editor] Cue points:', result.cuePoints);
            renderEditorCuePoints();
        } else if (result && result.error) {
            logger.warn('[Editor] Failed to load cue points:', result.error);
        } else {
            logger.warn('[Editor] Unexpected result format:', result);
        }
    } catch (error) {
        logger.warn('[Editor] Error loading cue points:', error);
        logger.warn('[Editor] Error stack:', error.stack);
    }
}

function handleEditorTimeUpdate() {
    if (!editorState || !editorState.elements || !editorState.elements.videoElement) {
        return;
    }

    const { videoElement, timelineElement, currentTimeLabel } = editorState.elements;
    const currentTime = Number(videoElement.currentTime) || 0;

    if (currentTimeLabel) {
        currentTimeLabel.textContent = formatTimestamp(currentTime, 2);
    }

    if (timelineElement && !editorState.isScrubbing) {
        timelineElement.value = currentTime.toFixed(3);
    }

    updateTimelinePlayhead(currentTime);
    handleEditorTimeUpdateLoopCheck();
}

function handleEditorScrubInput(event) {
    if (!editorState || !editorState.elements || !editorState.elements.videoElement) {
        return;
    }

    editorState.isScrubbing = true;
    const value = parseFloat(event.target.value);
    if (!Number.isFinite(value)) {
        return;
    }

    seekEditorTo(value);
}

function handleEditorScrubEnd() {
    if (editorState) {
        editorState.isScrubbing = false;
    }
}

function seekEditorTo(time) {
    if (!editorState || !editorState.elements) {
        return;
    }

    const duration = Number(editorState.duration) || 0;
    const clamped = Math.min(Math.max(time, 0), duration);
    const { videoElement, timelineElement, currentTimeLabel } = editorState.elements;

    if (videoElement && Math.abs(videoElement.currentTime - clamped) > 0.01) {
        videoElement.currentTime = clamped;
    } else if (videoElement) {
        videoElement.currentTime = clamped;
    }

    if (timelineElement) {
        timelineElement.value = clamped.toFixed(3);
    }

    if (currentTimeLabel) {
        currentTimeLabel.textContent = formatTimestamp(clamped, 2);
    }

    updateTimelinePlayhead(clamped);
}

function updateTimelinePlayhead(time) {
    if (!editorState || !editorState.elements || !editorState.elements.timelinePlayhead) {
        return;
    }

    const duration = Number(editorState.duration) || 0;
    const playhead = editorState.elements.timelinePlayhead;

    if (!Number.isFinite(duration) || duration <= 0) {
        playhead.style.left = '0%';
        return;
    }

    const percent = Math.min(100, Math.max(0, (time / duration) * 100));
    playhead.style.left = `${percent}%`;
}

function renderEditorTimelineScale() {
    if (!editorState || !editorState.elements || !editorState.elements.timelineScale) {
        return;
    }

    const { timelineScale } = editorState.elements;
    timelineScale.innerHTML = '';

    const duration = Number(editorState.duration) || 0;
    if (!Number.isFinite(duration) || duration <= 0) {
        return;
    }

    let step = 1;
    if (duration > 600) {
        step = 60;
    } else if (duration > 300) {
        step = 30;
    } else if (duration > 180) {
        step = 20;
    } else if (duration > 120) {
        step = 15;
    } else if (duration > 60) {
        step = 10;
    } else if (duration > 30) {
        step = 5;
    } else if (duration > 15) {
        step = 2;
    }

    for (let time = 0; time <= duration + 0.001; time += step) {
        const mark = document.createElement('div');
        const safeTime = Math.min(time, duration);
        const percent = Math.min(100, Math.max(0, (safeTime / duration) * 100));
        mark.className = 'timeline-scale-mark';
        mark.style.left = `${percent}%`;

        const line = document.createElement('span');
        line.className = 'timeline-scale-line';

        const label = document.createElement('span');
        label.className = 'timeline-scale-label';
        label.textContent = formatTimestamp(safeTime, duration >= 3600 ? 0 : 2);

        mark.appendChild(line);
        mark.appendChild(label);
        timelineScale.appendChild(mark);
    }
}

function renderEditorTimelineSegments() {
    if (!editorState || !editorState.elements || !editorState.elements.timelineSegmentsElement) {
        return;
    }

    const { timelineSegmentsElement } = editorState.elements;
    timelineSegmentsElement.innerHTML = '';

    const duration = Number(editorState.duration) || 0;
    if (!Number.isFinite(duration) || duration <= 0) {
        return;
    }

    (editorState.segments || []).forEach((segment, index) => {
        const widthPercent = ((segment.end - segment.start) / duration) * 100;
        const startPercent = (segment.start / duration) * 100;

        const segmentDiv = document.createElement('div');
        segmentDiv.className = `timeline-segment${segment.enabled ? '' : ' disabled'}`;
        segmentDiv.style.left = `${startPercent}%`;
        segmentDiv.style.width = `${Math.max(widthPercent, 0.4)}%`;
        segmentDiv.dataset.index = String(index);
        segmentDiv.title = `Segment ${index + 1}: ${formatTimestamp(segment.start, 2)} – ${formatTimestamp(segment.end, 2)}`;

        timelineSegmentsElement.appendChild(segmentDiv);

        segmentDiv.addEventListener('click', (event) => {
            event.stopPropagation();
            const rect = segmentDiv.getBoundingClientRect();
            const ratio = rect.width > 0 ? Math.min(Math.max((event.clientX - rect.left) / rect.width, 0), 1) : 0;
            const newTime = segment.start + ratio * (segment.end - segment.start);
            editorState.isScrubbing = false;
            seekEditorTo(newTime);
        });

        segmentDiv.addEventListener('dblclick', (event) => {
            event.stopPropagation();
            toggleSegmentEnabled(index);
        });

        const checkboxContainer = document.createElement('div');
        checkboxContainer.className = 'timeline-segment-checkbox-container';

        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.className = 'timeline-segment-checkbox';
        checkbox.checked = Boolean(segment.enabled);
        checkbox.title = segment.enabled ? 'Segment included' : 'Segment excluded';

        checkbox.addEventListener('click', (event) => {
            event.stopPropagation();
        });

        checkbox.addEventListener('change', (event) => {
            event.stopPropagation();
            setSegmentEnabled(index, checkbox.checked);
        });

        checkboxContainer.addEventListener('click', (event) => {
            event.stopPropagation();
        });

        checkboxContainer.appendChild(checkbox);
        segmentDiv.appendChild(checkboxContainer);
    });
}

function renderEditorCuePoints() {
    logger.log('[Editor] renderEditorCuePoints called');
    if (!editorState || !editorState.elements || !editorState.elements.timelineTrack) {
        logger.warn('[Editor] Cannot render cue points: editorState or elements missing');
        return;
    }

    // Get or create cue points container
    let cuePointsContainer = document.getElementById('editorTimelineCuePoints');
    if (!cuePointsContainer) {
        logger.warn('[Editor] Cue points container not found - timeline may not be initialized yet');
        return;
    }

    logger.log('[Editor] Cue points container found, clearing...');
    cuePointsContainer.innerHTML = '';

    const duration = Number(editorState.duration) || 0;
    logger.log('[Editor] Video duration:', duration);
    if (!Number.isFinite(duration) || duration <= 0) {
        logger.warn('[Editor] Invalid duration, cannot render cue points');
        return;
    }

    const cuePoints = editorState.cuePoints || [];
    logger.log('[Editor] Cue points to render:', cuePoints.length);
    if (cuePoints.length === 0) {
        logger.log('[Editor] No cue points to render');
        return;
    }

    // Render each cue point as a vertical marker
    logger.log('[Editor] Rendering cue point markers...');
    cuePoints.forEach((cuePoint, index) => {
        const startTime = Number(cuePoint.startTime) || 0;
        if (startTime < 0 || startTime > duration) {
            logger.warn(`[Editor] Skipping invalid cue point ${index}: startTime=${startTime}, duration=${duration}`);
            return; // Skip invalid cue points
        }

        const percent = Math.min(100, Math.max(0, (startTime / duration) * 100));
        logger.log(`[Editor] Rendering cue point ${index + 1}: "${cuePoint.title}" at ${startTime}s (${percent}%)`);

        const markerDiv = document.createElement('div');
        markerDiv.className = 'timeline-cue-point' + (cuePoint.source === 'audio-scan' ? ' audio-highlight' : '');
        markerDiv.style.left = `${percent}%`;
        markerDiv.dataset.index = String(index);
        
        const endTimeStr = cuePoint.endTime !== null && cuePoint.endTime !== undefined 
            ? formatTimestamp(cuePoint.endTime, 2) 
            : 'end';
        markerDiv.title = `${cuePoint.title || `Cue Point ${index + 1}`} (${formatTimestamp(startTime, 2)} – ${endTimeStr})`;

        // Add click handler to seek to cue point
        markerDiv.addEventListener('click', (event) => {
            event.stopPropagation();
            editorState.isScrubbing = false;
            seekEditorTo(startTime);
        });

        cuePointsContainer.appendChild(markerDiv);
    });
    logger.log('[Editor] Finished rendering', cuePoints.length, 'cue point markers');
}

function setSegmentEnabled(index, enabled) {
    if (!editorState || !Array.isArray(editorState.segments)) {
        return;
    }

    const segment = editorState.segments[index];
    if (!segment) {
        return;
    }

    const nextEnabled = Boolean(enabled);
    if (segment.enabled === nextEnabled) {
        return;
    }

    editorState.segmentStates.set(segment.key, nextEnabled);
    editorState.segments[index] = { ...segment, enabled: nextEnabled };
    updateSegmentSummaryAndTimeline();
    updateEditorControlsAvailability();
    setEditorStatus(nextEnabled ? `Segment ${index + 1} included.` : `Segment ${index + 1} excluded from export.`, nextEnabled ? 'success' : 'info');
}

function toggleSegmentEnabled(index) {
    if (!editorState || !Array.isArray(editorState.segments)) {
        return;
    }

    const segment = editorState.segments[index];
    if (!segment) {
        return;
    }

    setSegmentEnabled(index, !segment.enabled);
}

function updateEditorControlsAvailability() {
    if (!editorState || !editorState.elements) {
        return;
    }

    const hasDuration = Number.isFinite(editorState.duration) && editorState.duration > 0;
    const hasAdditionalSplits = Array.isArray(editorState.boundaries) && editorState.boundaries.length > 2;

    if (editorState.elements.splitBtn) {
        editorState.elements.splitBtn.disabled = !hasDuration || editorState.exporting;
    }
    if (editorState.elements.resetSplitsBtn) {
        editorState.elements.resetSplitsBtn.disabled = !hasDuration || editorState.exporting || !hasAdditionalSplits;
    }
    if (editorState.elements.exportBtn) {
        const hasActiveSegments = editorState.segments && editorState.segments.some(segment => segment.enabled);
        editorState.elements.exportBtn.disabled = !hasDuration || editorState.exporting || !hasActiveSegments;
    }
    updateHighlightDetectUI();
}

function setEditorStatus(message = '', type = 'info') {
    if (!editorState || !editorState.elements || !editorState.elements.statusElement) {
        return;
    }

    const statusEl = editorState.elements.statusElement;
    statusEl.textContent = message;
    statusEl.classList.remove('info', 'success', 'error');
    if (message) {
        statusEl.classList.add(type);
    }
}

function updateHighlightDetectUI() {
    if (!editorState || !editorState.elements) return;
    const { highlightDetectRange, highlightScanBtn, highlightSaveTemplateBtn, highlightDeleteTemplateBtn, highlightClearBtn, highlightPlayLoopBtn } = editorState.elements;
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;
    const windowSec = windowMs / 1000;
    const hasValidMarker = marker != null && marker >= 0;
    const hasTemplate = editorState.currentTemplate != null;
    if (highlightDetectRange) {
        if (hasTemplate) {
            highlightDetectRange.textContent = `Template: ${editorState.currentTemplate.name}`;
        } else if (hasValidMarker) {
            const start = marker - windowSec / 2;
            const end = marker + windowSec / 2;
            highlightDetectRange.textContent = `${formatTimestamp(start, 2)} – ${formatTimestamp(end, 2)} (${windowMs}ms)`;
        } else {
            highlightDetectRange.textContent = 'Click waveform to set marker';
        }
    }
    if (highlightScanBtn) {
        highlightScanBtn.disabled = (hasTemplate ? false : !hasValidMarker) || editorState.highlightScanning || editorState.exporting;
    }
    if (highlightSaveTemplateBtn) {
        const hasVisualTemplate = editorState.visualTemplateImage != null;
        highlightSaveTemplateBtn.disabled = (!hasValidMarker && !hasVisualTemplate) || editorState.highlightScanning || editorState.exporting;
    }
    if (highlightDeleteTemplateBtn) {
        highlightDeleteTemplateBtn.style.display = hasTemplate ? 'inline-block' : 'none';
    }
    if (highlightExportTemplateBtn) {
        highlightExportTemplateBtn.style.display = hasTemplate ? 'inline-block' : 'none';
    }
    if (highlightPlayLoopBtn) {
        highlightPlayLoopBtn.disabled = !hasValidMarker;
        highlightPlayLoopBtn.textContent = editorState.templateLoopPlaying ? 'Stop loop' : 'Play loop';
    }
    const hasAudioMarkers = (editorState.cuePoints || []).some(cp => cp.source === 'audio-scan');
    if (highlightClearBtn) {
        highlightClearBtn.disabled = !hasAudioMarkers;
    }
    updateHighlightWaveformMarkers();
}

function updateHighlightWaveformMarkers() {
    if (!editorState || !editorState.elements) return;
    const { highlightMarker, highlightWaveformRange } = editorState.elements;
    if (editorState.currentTemplate) {
        if (highlightMarker) highlightMarker.style.display = 'none';
        if (highlightWaveformRange) highlightWaveformRange.style.display = 'none';
        return;
    }
    const duration = Number(editorState.duration) || 0;
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;
    if (!duration) return;
    const windowSec = windowMs / 1000;
    if (marker != null && marker >= 0 && marker <= duration) {
        const markerPercent = (marker / duration) * 100;
        if (highlightMarker) {
            highlightMarker.style.left = `${markerPercent}%`;
            highlightMarker.style.display = 'block';
        }
        if (highlightWaveformRange) {
            const startSec = Math.max(0, marker - windowSec / 2);
            const endSec = Math.min(duration, marker + windowSec / 2);
            const startPercent = (startSec / duration) * 100;
            const endPercent = (endSec / duration) * 100;
            highlightWaveformRange.style.left = `${startPercent}%`;
            highlightWaveformRange.style.width = `${endPercent - startPercent}%`;
            highlightWaveformRange.style.display = 'block';
        }
    } else {
        if (highlightMarker) highlightMarker.style.display = 'none';
        if (highlightWaveformRange) highlightWaveformRange.style.display = 'none';
    }
}

function drawHighlightWaveform() {
    if (!editorState || !editorState.elements || !editorState.waveformPeaks) return;
    const canvas = editorState.elements.highlightWaveformCanvas;
    const container = canvas && canvas.parentElement;
    if (!canvas || !container) return;
    const peaks = editorState.waveformPeaks;
    const w = container.clientWidth || 800;
    const h = container.clientHeight || 80;
    canvas.width = w;
    canvas.height = h;
    const ctx = canvas.getContext('2d');
    const midY = h / 2;
    ctx.fillStyle = '#1a1a1a';
    ctx.fillRect(0, 0, w, h);
    ctx.fillStyle = 'rgba(0, 149, 222, 0.5)';
    const barW = Math.max(1, w / peaks.length);
    for (let i = 0; i < peaks.length; i++) {
        const amp = peaks[i] * midY * 0.9;
        const x = (i / peaks.length) * w;
        ctx.fillRect(x, midY - amp, barW + 1, amp * 2);
    }
}

function setupHighlightWaveformInteraction() {
    if (!editorState || !editorState.elements) return;
    const overlay = editorState.elements.highlightWaveformOverlay;
    if (!overlay) return;

    const SCRUB_CLICK_THRESHOLD = 5;
    let draggingMarker = false;
    let waveformScrubbing = false;
    let scrubDownX = 0;

    function xToTime(clientX) {
        const rect = overlay.getBoundingClientRect();
        const duration = Number(editorState.duration) || 0;
        if (!duration) return 0;
        const x = clientX - rect.left;
        const ratio = Math.max(0, Math.min(1, x / rect.width));
        return ratio * duration;
    }

    function setMarkerAt(clientX) {
        const tc = Number(xToTime(clientX).toFixed(3));
        editorState.templateMarker = tc;
        updateHighlightDetectUI();
        seekEditorTo(tc);
    }

    overlay.addEventListener('mousedown', (e) => {
        if (!editorState || !editorState.duration) return;
        if (editorState.currentTemplate) return;
        draggingMarker = true;
        waveformScrubbing = true;
        scrubDownX = e.clientX;
        const tc = xToTime(e.clientX);
        editorState.templateMarker = tc;
        updateHighlightDetectUI();
        seekEditorTo(tc);
        const { videoElement } = editorState.elements;
        if (videoElement) videoElement.play().catch(() => {});
    });

    window.addEventListener('mousemove', (e) => {
        if (!editorState) return;
        if (draggingMarker || waveformScrubbing) {
            const tc = xToTime(e.clientX);
            editorState.templateMarker = tc;
            updateHighlightDetectUI();
            seekEditorTo(tc);
            const { videoElement } = editorState.elements;
            if (videoElement) videoElement.play().catch(() => {});
        }
    });

    window.addEventListener('mouseup', (e) => {
        if (draggingMarker) {
            draggingMarker = false;
        }
        if (waveformScrubbing) {
            waveformScrubbing = false;
            const { videoElement } = editorState.elements;
            if (videoElement) videoElement.pause();
            if (Math.abs(e.clientX - scrubDownX) < SCRUB_CLICK_THRESHOLD) {
                setMarkerAt(e.clientX);
            }
        }
    });
}

async function loadHighlightWaveform() {
    if (!editorState || !editorState.video || !editorState.duration) return;
    const loading = editorState.elements.highlightWaveformLoading;
    if (loading) loading.style.display = 'block';
    try {
        const result = await ipcRenderer.invoke('get-waveform-data', {
            videoPath: editorState.video.path,
            duration: editorState.duration
        });
        if (result && result.success && Array.isArray(result.peaks)) {
            editorState.waveformPeaks = result.peaks;
            drawHighlightWaveform();
            updateHighlightWaveformMarkers();
        }
    } catch (err) {
        logger.warn('[Editor] Waveform load error:', err);
    }
    if (loading) loading.style.display = 'none';

    window.addEventListener('resize', function onWaveformResize() {
        if (!editorState || editorState !== state.editorState) {
            window.removeEventListener('resize', onWaveformResize);
            return;
        }
        if (editorState.waveformPeaks) drawHighlightWaveform();
    });
}

function toggleTemplateLoopPlayback() {
    if (!editorState || !editorState.elements) return;
    const { videoElement } = editorState.elements;
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;
    if (marker == null) return;

    editorState.templateLoopPlaying = !editorState.templateLoopPlaying;
    updateHighlightDetectUI();

    if (editorState.templateLoopPlaying) {
        const start = marker - windowMs / 2000;
        seekEditorTo(start);
        videoElement.play().catch(() => {});
    } else {
        videoElement.pause();
    }
}

function handleEditorTimeUpdateLoopCheck() {
    if (!editorState || !editorState.templateLoopPlaying || !editorState.elements) return;
    const { videoElement } = editorState.elements;
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;
    if (marker == null) return;
    const start = marker - windowMs / 2000;
    const end = marker + windowMs / 2000;
    const t = Number(videoElement.currentTime) || 0;
    if (t >= end - 0.02) {
        seekEditorTo(start);
    }
}

function clearAudioHighlightMarkers() {
    if (!editorState || !editorState.cuePoints) return;
    editorState.cuePoints = editorState.cuePoints.filter(cp => cp.source !== 'audio-scan');
    renderEditorCuePoints();
    updateHighlightDetectUI();
    setEditorStatus('Cleared detected highlight markers.', 'info');
}

async function populateHighlightTemplates() {
    const select = editorState?.elements?.highlightTemplateSelect;
    if (!select) return;
    try {
        const result = await ipcRenderer.invoke('load-highlight-templates');
        const templates = result?.templates || [];
        const currentVal = select.value;
        select.innerHTML = '<option value="">— Select template —</option>';
        for (const t of templates) {
            const opt = document.createElement('option');
            opt.value = t.name;
            opt.textContent = t.name;
            select.appendChild(opt);
        }
        if (currentVal && templates.some(t => t.name === currentVal)) {
            select.value = currentVal;
        }
    } catch (err) {
        logger.warn('[Editor] Failed to load highlight templates:', err);
    }
}

function openSaveTemplateModal() {
    if (!editorState || !saveTemplateModal || !saveTemplateInput) return;
    const marker = editorState.templateMarker;
    const hasVisualTemplate = editorState.visualTemplateImage != null;
    // Allow opening modal if we have either an audio marker or a visual template
    if (marker == null && !hasVisualTemplate) return;
    saveTemplateInput.value = '';
    if (saveTemplateError) saveTemplateError.textContent = '';
    if (saveTemplateConfirmBtn) saveTemplateConfirmBtn.disabled = false;
    saveTemplateModal.classList.add('active');
    setTimeout(() => {
        saveTemplateInput.focus();
    }, 0);
}

function closeSaveTemplateModal() {
    if (!saveTemplateModal) return;
    saveTemplateModal.classList.remove('active');
}

function setSaveTemplateError(msg) {
    if (saveTemplateError) saveTemplateError.textContent = msg || '';
}

async function submitSaveTemplate() {
    if (!editorState || !saveTemplateInput) return;
    const name = saveTemplateInput.value.trim();
    if (!name) {
        setSaveTemplateError('Template name cannot be empty.');
        saveTemplateInput.focus();
        return;
    }
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;
    const hasVisualTemplate = editorState.visualTemplateImage != null;

    // If no marker, we can only save if we have a visual template
    if (marker == null) {
        if (!hasVisualTemplate) {
            setSaveTemplateError('Set an audio marker or extract a visual template first.');
            return;
        }
    } else if (windowMs < 100) {
        setSaveTemplateError('Invalid marker or window.');
        return;
    } else {
        // Validate window when marker exists
        const templateStart = marker - windowMs / 2000;
        const templateEnd = marker + windowMs / 2000;
        if (templateStart < 0 || templateEnd > editorState.duration || templateEnd - templateStart < 0.3) {
            setSaveTemplateError('Template window extends beyond video or is too short.');
            return;
        }
    }

    try {
        if (saveTemplateConfirmBtn) saveTemplateConfirmBtn.disabled = true;
        setSaveTemplateError('');

        let audioSample = null;
        let templateType = hasVisualTemplate ? 'visual' : 'audio';

        // Only extract audio if marker is set
        if (marker != null) {
            const templateStart = marker - windowMs / 2000;
            const templateEnd = marker + windowMs / 2000;
            const extractResult = await ipcRenderer.invoke('extract-audio-sample', {
                videoPath: editorState.video.path,
                templateStart,
                templateEnd
            });
            if (!extractResult?.success || !extractResult.audioSample) {
                setSaveTemplateError(extractResult?.error || 'Failed to extract audio sample.');
                if (saveTemplateConfirmBtn) saveTemplateConfirmBtn.disabled = false;
                return;
            }
            audioSample = extractResult.audioSample;
            // Determine type based on what we have
            templateType = hasVisualTemplate ? 'visual' : 'audio';
        }

        const saveResult = await ipcRenderer.invoke('save-highlight-template', {
            name,
            audioSample,
            threshold: editorState.templateThreshold,
            minGap: editorState.templateMinGap,
            visualTemplate: hasVisualTemplate ? {
                imageData: editorState.visualTemplateImage,
                region: editorState.visualRegion,
                method: editorState.visualMethod || 'gpu',
                timestamp: editorState.visualRegion?.timestamp || 0,
                frameSkip: editorState.visualFrameSkip || 5,
                batchSize: editorState.visualBatchSize || 128,
                minGap: parseFloat(document.getElementById('visualMinGap')?.value) || editorState.visualMinGap || 1
            } : null,
            visualThreshold: editorState.visualThreshold || 0.7,
            type: templateType
        });
        if (!saveResult?.success) {
            setSaveTemplateError(saveResult?.error || 'Failed to save template.');
            if (saveTemplateConfirmBtn) saveTemplateConfirmBtn.disabled = false;
            return;
        }
        closeSaveTemplateModal();
        setEditorStatus(`Template "${name}" saved.`, 'success');
        populateHighlightTemplates();
    } catch (err) {
        logger.warn('[Editor] Save template error:', err);
        setSaveTemplateError(err?.message || 'Failed to save template.');
        if (saveTemplateConfirmBtn) saveTemplateConfirmBtn.disabled = false;
    }
}

async function onHighlightTemplateSelectChange() {
    const select = editorState?.elements?.highlightTemplateSelect;
    if (!select || !editorState) return;
    const name = select.value;
    if (!name) {
        editorState.currentTemplate = null;
        updateHighlightDetectUI();
        return;
    }
    try {
        const result = await ipcRenderer.invoke('load-highlight-template-by-name', name);
        if (!result?.success || !result.template) {
            setEditorStatus(result?.error || 'Failed to load template.', 'error');
            return;
        }
        editorState.currentTemplate = result.template;
        const { highlightThreshold, highlightMinGap } = editorState.elements;
        if (highlightThreshold) {
            highlightThreshold.value = String(editorState.currentTemplate.threshold);
            highlightThreshold.disabled = false;
            highlightThreshold.readOnly = false;
        }
        if (highlightMinGap) {
            highlightMinGap.value = String(editorState.currentTemplate.minGap);
            highlightMinGap.disabled = false;
            highlightMinGap.readOnly = false;
        }
        editorState.templateThreshold = editorState.currentTemplate.threshold;
        editorState.templateMinGap = editorState.currentTemplate.minGap;
        
        // Restore visual template data if present
        if (editorState.currentTemplate.visualTemplate) {
            editorState.visualTemplateImage = editorState.currentTemplate.visualTemplate.imageData;
            editorState.visualRegion = editorState.currentTemplate.visualTemplate.region;
            editorState.visualMethod = editorState.currentTemplate.visualTemplate.method || 'gpu';
            editorState.visualFrameSkip = editorState.currentTemplate.visualTemplate.frameSkip || 5;
            editorState.visualBatchSize = editorState.currentTemplate.visualTemplate.batchSize || 128;
            editorState.visualMinGap = editorState.currentTemplate.visualTemplate.minGap || 1;
            editorState.visualThreshold = editorState.currentTemplate.visualThreshold || 0.7;

            // Update UI inputs
            const visualMethodSelect = document.getElementById('visualMethodSelect');
            if (visualMethodSelect) {
                visualMethodSelect.value = editorState.visualMethod || 'gpu';
                visualMethodSelect.disabled = false;
            }
            const visualThresholdInput = document.getElementById('visualThreshold');
            if (visualThresholdInput) {
                visualThresholdInput.value = String(editorState.visualThreshold);
                visualThresholdInput.disabled = false;
                visualThresholdInput.readOnly = false;
            }
            const visualFrameSkipInput = document.getElementById('visualFrameSkip');
            if (visualFrameSkipInput) {
                visualFrameSkipInput.value = String(editorState.visualFrameSkip);
                visualFrameSkipInput.disabled = false;
                visualFrameSkipInput.readOnly = false;
            }
            const visualBatchSizeInput = document.getElementById('visualBatchSize');
            if (visualBatchSizeInput) {
                visualBatchSizeInput.value = String(editorState.visualBatchSize);
                visualBatchSizeInput.disabled = false;
                visualBatchSizeInput.readOnly = false;
            }
            const visualMinGapInput = document.getElementById('visualMinGap');
            if (visualMinGapInput) {
                visualMinGapInput.value = String(editorState.visualMinGap);
                visualMinGapInput.disabled = false;
                visualMinGapInput.readOnly = false;
            }
            
            // Show visual template preview
            const visualTemplatePreview = document.getElementById('visualTemplatePreview');
            const visualTemplateImage = document.getElementById('visualTemplateImage');
            if (visualTemplatePreview && visualTemplateImage) {
                visualTemplatePreview.style.display = 'flex';
                visualTemplateImage.src = `data:image/png;base64,${editorState.visualTemplateImage}`;
            }
            
            // Enable visual scan button when template loaded
            const visualScanBtn = document.getElementById('visualScanBtn');
            if (visualScanBtn) { visualScanBtn.disabled = false; visualScanBtn.title = 'Scan video for visual matches'; }
            
            // Switch to appropriate detection mode
            if (editorState.currentTemplate.type === 'visual') {
                const visualModeTab = document.getElementById('visualModeTab');
                const visualPanel = document.getElementById('visualPanel');
                const audioModeTab = document.getElementById('audioModeTab');
                const audioPanel = document.getElementById('audioPanel');
                editorState.detectionMode = 'visual';
                if (audioModeTab) audioModeTab.classList.remove('active');
                if (visualModeTab) visualModeTab.classList.add('active');
                if (audioPanel) audioPanel.classList.remove('active');
                if (visualPanel) visualPanel.classList.add('active');
            }
        }
        
        updateHighlightDetectUI();
        setEditorStatus(`Template "${name}" loaded. Click Scan to find matches.`, 'info');
    } catch (err) {
        logger.warn('[Editor] Load template error:', err);
        setEditorStatus(err?.message || 'Failed to load template.', 'error');
    }
}

async function deleteSelectedTemplate() {
    if (!editorState || !editorState.currentTemplate) return;
    const name = editorState.currentTemplate.name;
    const confirmed = confirm(`Delete template "${name}"? This cannot be undone.`);
    if (!confirmed) return;

    try {
        const result = await ipcRenderer.invoke('delete-highlight-template', name);
        if (!result?.success) {
            setEditorStatus(result?.error || 'Failed to delete template.', 'error');
            return;
        }
        editorState.currentTemplate = null;
        const { highlightTemplateSelect, highlightThreshold, highlightMinGap } = editorState.elements;
        if (highlightTemplateSelect) highlightTemplateSelect.value = '';
        updateHighlightDetectUI();
        populateHighlightTemplates();
        setEditorStatus(`Template "${name}" deleted.`, 'info');
    } catch (err) {
        logger.warn('[Editor] Delete template error:', err);
        setEditorStatus(err?.message || 'Failed to delete template.', 'error');
    }
}

async function exportSelectedTemplate() {
    if (!editorState || !editorState.currentTemplate) return;
    const name = editorState.currentTemplate.name;

    try {
        const { filePath } = await ipcRenderer.invoke('show-save-dialog', {
            title: 'Export Highlight Template',
            defaultPath: `${name}.zip`,
            filters: [
                { name: 'ZIP Files', extensions: ['zip'] }
            ]
        });

        if (!filePath) return;

        const result = await ipcRenderer.invoke('export-highlight-template', {
            name,
            outputPath: filePath
        });

        if (!result?.success) {
            setEditorStatus(result?.error || 'Failed to export template.', 'error');
            return;
        }

        setEditorStatus(`Template "${name}" exported successfully.`, 'info');
    } catch (err) {
        logger.warn('[Editor] Export template error:', err);
        setEditorStatus(err?.message || 'Failed to export template.', 'error');
    }
}

async function importTemplate() {
    try {
        const { filePaths } = await ipcRenderer.invoke('show-open-dialog', {
            title: 'Import Highlight Template',
            filters: [
                { name: 'ZIP Files', extensions: ['zip'] }
            ],
            properties: ['openFile']
        });

        if (!filePaths || filePaths.length === 0) return;

        const result = await ipcRenderer.invoke('import-highlight-template', filePaths[0]);

        if (!result?.success) {
            setEditorStatus(result?.error || 'Failed to import template.', 'error');
            return;
        }

        populateHighlightTemplates();
        setEditorStatus(result.message || `Template "${result.template?.name}" imported.`, 'info');
    } catch (err) {
        logger.warn('[Editor] Import template error:', err);
        setEditorStatus(err?.message || 'Failed to import template.', 'error');
    }
}

async function runHighlightDetection() {
    if (!editorState || editorState.highlightScanning || !editorState.video) return;
    const hasTemplate = editorState.currentTemplate != null;
    const marker = editorState.templateMarker;
    const windowMs = editorState.templateWindowMs || 1500;

    if (hasTemplate) {
        editorState.highlightScanning = true;
        updateHighlightDetectUI();
        const { highlightDetectProgress, highlightDetectProgressFill, highlightDetectProgressText } = editorState.elements;
        if (highlightDetectProgress) highlightDetectProgress.style.display = 'flex';
        if (highlightDetectProgressText) highlightDetectProgressText.textContent = 'Starting scan...';
        if (highlightDetectProgressFill) highlightDetectProgressFill.style.width = '0%';
        setEditorStatus('Scanning for audio matches...', 'info');

        try {
            const result = await ipcRenderer.invoke('detect-audio-highlights-with-template', {
                videoPath: editorState.video.path,
                audioSampleBase64: editorState.currentTemplate.audioSample,
                similarityThreshold: editorState.templateThreshold,
                minGapSeconds: editorState.templateMinGap
            });

            if (result && result.success && Array.isArray(result.matches)) {
                const existing = editorState.cuePoints || [];
                const nonAudio = existing.filter(cp => cp.source !== 'audio-scan');
                editorState.cuePoints = [...nonAudio, ...result.matches];
                renderEditorCuePoints();
                setEditorStatus(`Found ${result.matches.length} highlight(s).`, 'success');
            } else {
                setEditorStatus(result?.error || 'No matches found.', result?.success ? 'info' : 'error');
            }
        } catch (err) {
            logger.warn('[Editor] Highlight detection error:', err);
            setEditorStatus(err?.message || 'Highlight detection failed.', 'error');
        } finally {
            editorState.highlightScanning = false;
            updateHighlightDetectUI();
            if (highlightDetectProgress) highlightDetectProgress.style.display = 'none';
        }
        return;
    }

    if (marker == null || windowMs < 100) {
        setEditorStatus('Set a valid template marker and window or load a template.', 'error');
        return;
    }

    const templateStart = marker - windowMs / 2000;
    const templateEnd = marker + windowMs / 2000;
    if (templateStart < 0 || templateEnd > editorState.duration || templateEnd - templateStart < 0.3) {
        setEditorStatus('Template window extends beyond video or is too short.', 'error');
        return;
    }

    editorState.highlightScanning = true;
    updateHighlightDetectUI();
    const { highlightDetectProgress, highlightDetectProgressFill, highlightDetectProgressText } = editorState.elements;
    if (highlightDetectProgress) highlightDetectProgress.style.display = 'flex';
    if (highlightDetectProgressText) highlightDetectProgressText.textContent = 'Starting scan...';
    if (highlightDetectProgressFill) highlightDetectProgressFill.style.width = '0%';
    setEditorStatus('Scanning for audio matches...', 'info');

    try {
        const result = await ipcRenderer.invoke('detect-audio-highlights', {
            videoPath: editorState.video.path,
            templateStart: templateStart,
            templateEnd: templateEnd,
            similarityThreshold: editorState.templateThreshold,
            minGapSeconds: editorState.templateMinGap
        });

        if (result && result.success && Array.isArray(result.matches)) {
            const existing = editorState.cuePoints || [];
            const nonAudio = existing.filter(cp => cp.source !== 'audio-scan');
            editorState.cuePoints = [...nonAudio, ...result.matches];
            renderEditorCuePoints();
            setEditorStatus(`Found ${result.matches.length} highlight(s).`, 'success');
        } else {
            setEditorStatus(result?.error || 'No matches found.', result?.success ? 'info' : 'error');
        }
    } catch (err) {
        logger.warn('[Editor] Highlight detection error:', err);
        setEditorStatus(err?.message || 'Highlight detection failed.', 'error');
    } finally {
        editorState.highlightScanning = false;
        updateHighlightDetectUI();
        if (highlightDetectProgress) highlightDetectProgress.style.display = 'none';
    }
}

async function runVisualDetection() {
    if (!editorState || editorState.highlightScanning || !editorState.video) return;
    if (!editorState.visualRegion) {
        setEditorStatus('Select a region on the video first.', 'error');
        return;
    }
    if (!editorState.visualTemplateImage) {
        setEditorStatus('Extract a visual template from the selected region first.', 'error');
        return;
    }

    editorState.highlightScanning = true;
    updateHighlightDetectUI();
    const { highlightDetectProgress, highlightDetectProgressFill, highlightDetectProgressText } = editorState.elements;
    if (highlightDetectProgress) highlightDetectProgress.style.display = 'flex';
    if (highlightDetectProgressText) highlightDetectProgressText.textContent = 'Starting visual scan...';
    if (highlightDetectProgressFill) highlightDetectProgressFill.style.width = '0%';
    setEditorStatus('Scanning for visual matches...', 'info');

    try {
        const result = await ipcRenderer.invoke('detect-visual-highlights', {
            videoPath: editorState.video.path,
            templateBase64: editorState.visualTemplateImage,
            region: editorState.visualRegion,
            method: editorState.visualMethod || 'gpu',
            threshold: editorState.visualThreshold || 0.7,
            frameSkip: editorState.visualFrameSkip || 5,
            batchSize: editorState.visualBatchSize || 128,
            minGapSeconds: parseFloat(document.getElementById('visualMinGap')?.value) || editorState.visualMinGap || 1
        });

        if (result && result.success && Array.isArray(result.matches)) {
            const existing = editorState.cuePoints || [];
            const nonVisual = existing.filter(cp => cp.source !== 'visual-scan');
            editorState.cuePoints = [...nonVisual, ...result.matches];
            renderEditorCuePoints();
            setEditorStatus(`Found ${result.matches.length} visual highlight(s).`, 'success');
            
            if (result.frameResults && Array.isArray(result.frameResults)) {
                editorState.lastVisualScanResults = result.frameResults;
                const viewResultsBtn = document.getElementById('viewVisualResultsBtn');
                if (viewResultsBtn) { viewResultsBtn.disabled = false; viewResultsBtn.title = 'View scan results'; }
            }
        } else {
            setEditorStatus(result?.error || 'No matches found.', result?.success ? 'info' : 'error');
        }
    } catch (err) {
        logger.warn('[Editor] Visual detection error:', err);
        setEditorStatus(err?.message || 'Visual detection failed.', 'error');
    } finally {
        editorState.highlightScanning = false;
        updateHighlightDetectUI();
        if (highlightDetectProgress) highlightDetectProgress.style.display = 'none';
    }
}


async function extractVisualRegion() {
    if (!editorState || !editorState.video || !editorState.visualRegion) return;
    const videoElement = editorState.elements?.videoElement;
    if (!videoElement) return;

    const currentTime = videoElement.currentTime;
    setEditorStatus('Extracting visual template...', 'info');

    try {
        const result = await ipcRenderer.invoke('extract-visual-region', {
            videoPath: editorState.video.path,
            timestamp: currentTime,
            region: editorState.visualRegion
        });

        if (result && result.success && result.template) {
            editorState.visualTemplateImage = result.template.imageData;
            
            const visualTemplatePreview = document.getElementById('visualTemplatePreview');
            const visualTemplateImage = document.getElementById('visualTemplateImage');
            const visualTemplateMethod = document.getElementById('visualTemplateMethod');
            const visualScanBtn = document.getElementById('visualScanBtn');
            
            if (visualTemplatePreview) visualTemplatePreview.style.display = 'flex';
            if (visualTemplateImage) {
                visualTemplateImage.src = `data:image/png;base64,${result.template.imageData}`;
            }
            if (visualTemplateMethod) {
                const methodNames = { fast: 'Fast (Resize+MSE)', balanced: 'Balanced (pixelmatch)', accurate: 'Accurate (OpenCV)' };
                visualTemplateMethod.textContent = methodNames[editorState.visualMethod] || 'Fast';
            }
            if (visualScanBtn) { visualScanBtn.disabled = false; visualScanBtn.title = 'Scan video for visual matches'; }

            setEditorStatus('Visual template extracted. Ready to scan.', 'success');
        } else {
            setEditorStatus(result?.error || 'Failed to extract visual template.', 'error');
        }
    } catch (err) {
        logger.warn('[Editor] Visual region extraction error:', err);
        setEditorStatus(err?.message || 'Visual region extraction failed.', 'error');
    }
}

function addEditorSplit() {
    if (!editorState || !editorState.elements || !editorState.elements.videoElement) {
        return;
    }

    const duration = editorState.duration;
    if (!Number.isFinite(duration) || duration <= 0) {
        setEditorStatus('Video metadata not loaded yet.', 'error');
        return;
    }

    const currentTime = Number(editorState.elements.videoElement.currentTime) || 0;
    const tolerance = 0.05;

    if (currentTime <= tolerance || duration - currentTime <= tolerance) {
        setEditorStatus('Place split at least a few frames away from the start or end.', 'error');
        return;
    }

    const exists = (editorState.boundaries || []).some(boundary => Math.abs(boundary - currentTime) < tolerance);
    if (exists) {
        setEditorStatus('A split already exists near this position.', 'error');
        return;
    }

    editorState.boundaries.push(Number(currentTime.toFixed(3)));
    updateEditorSegments();
    setEditorStatus(`Added split at ${formatTimestamp(currentTime, 2)}.`, 'success');
}

function resetEditorSplits() {
    if (!editorState || !Number.isFinite(editorState.duration) || editorState.duration <= 0) {
        return;
    }

    editorState.boundaries = [0, Number(editorState.duration.toFixed(3))];
    editorState.segmentStates = new Map();
    updateEditorSegments();
    setEditorStatus('Splits reset to full video.', 'info');
}

function removeEditorBoundary(boundaryIndex) {
    if (!editorState || !Array.isArray(editorState.boundaries)) {
        return;
    }

    if (!Number.isInteger(boundaryIndex) && !Number.isFinite(boundaryIndex)) {
        return;
    }

    const index = Number(boundaryIndex);
    if (index <= 0 || index >= editorState.boundaries.length - 1) {
        return;
    }

    editorState.boundaries.splice(index, 1);
    updateEditorSegments();
    setEditorStatus('Removed split point.', 'info');
}

function updateEditorSegments() {
    if (!editorState) {
        return;
    }

    const duration = editorState.duration;
    if (!Number.isFinite(duration) || duration <= 0) {
        editorState.segments = [];
        updateSegmentSummaryAndTimeline();
        updateEditorControlsAvailability();
        return;
    }

    const unique = new Set();
    (editorState.boundaries || []).forEach(value => {
        if (Number.isFinite(value)) {
            const clamped = Math.min(Math.max(value, 0), duration);
            unique.add(Number(clamped.toFixed(3)));
        }
    });

    unique.add(0);
    unique.add(Number(duration.toFixed(3)));

    const sortedBoundaries = Array.from(unique).sort((a, b) => a - b);
    editorState.boundaries = sortedBoundaries;

    const previousStates = editorState.segmentStates instanceof Map ? editorState.segmentStates : new Map();
    const newSegments = [];
    const newStates = new Map();

    for (let i = 0; i < sortedBoundaries.length - 1; i += 1) {
        const start = sortedBoundaries[i];
        const end = sortedBoundaries[i + 1];

        if (end - start <= 0.05) {
            continue;
        }

        const key = `${start.toFixed(3)}-${end.toFixed(3)}`;
        const enabled = previousStates.has(key) ? previousStates.get(key) : true;
        newStates.set(key, enabled);

        newSegments.push({
            start,
            end,
            key,
            enabled,
            startBoundaryIndex: i,
            endBoundaryIndex: i + 1
        });
    }

    editorState.segmentStates = newStates;
    editorState.segments = newSegments;

    updateSegmentSummaryAndTimeline();
    updateEditorControlsAvailability();
}

function updateSegmentSummaryAndTimeline() {
    if (!editorState || !editorState.elements) {
        return;
    }

    const { segmentSummaryElement, exportBtn } = editorState.elements;
    const segments = Array.isArray(editorState.segments) ? editorState.segments : [];
    const includedSegments = segments.filter(segment => segment.enabled);

    if (segmentSummaryElement) {
        if (segments.length === 0) {
            segmentSummaryElement.textContent = 'No splits — click the scissor icon to add one.';
        } else {
            const allIncluded = includedSegments.length === segments.length;
            const summaryParts = [
                `${segments.length} segment${segments.length !== 1 ? 's' : ''}`,
                allIncluded ? 'all included' : `${includedSegments.length} included`,
                'double-click segments to toggle'
            ];
            segmentSummaryElement.textContent = summaryParts.join(' • ');
        }
    }

    if (exportBtn) {
        exportBtn.disabled = editorState.exporting || includedSegments.length === 0;
    }

    renderEditorTimelineSegments();
    renderEditorCuePoints();
}

async function exportEditedVideo() {
    if (!editorState || editorState.exporting) {
        return;
    }

    const enabledSegments = (editorState.segments || []).filter(segment => segment.enabled && segment.end - segment.start > 0.05);
    if (enabledSegments.length === 0) {
        setEditorStatus('Select at least one segment to export.', 'error');
        return;
    }

    editorState.exporting = true;
    setEditorStatus('Export in progress...', 'info');
    updateEditorControlsAvailability();

    try {
        const payload = {
            videoPath: editorState.video.path,
            segments: enabledSegments.map(segment => ({
                start: Number(segment.start.toFixed(3)),
                end: Number(segment.end.toFixed(3))
            }))
        };

        const result = await ipcRenderer.invoke('export-video-splits', payload);

        if (!result || result.canceled) {
            setEditorStatus('Export canceled.', 'info');
        } else if (!result.success) {
            setEditorStatus(result.error || 'Failed to export video.', 'error');
        } else {
            setEditorStatus(`Exported to ${result.outputPath}`, 'success');
            await scanAllDirectories();
        }
    } catch (error) {
        logger.error('Export error:', error);
        setEditorStatus('Export failed. Check the console for details.', 'error');
    } finally {
        editorState.exporting = false;
        updateEditorControlsAvailability();
        updateSegmentSummaryAndTimeline();
    }
}


function applyVideoRename(oldPath, updatedVideo) {
    const index = videos.findIndex(v => v.path === oldPath);
    if (index === -1) {
        return;
    }

    const mergedVideo = {
        ...videos[index],
        ...updatedVideo
    };

    videos[index] = mergedVideo;
    videoMap.delete(oldPath);
    videoMap.set(mergedVideo.path, mergedVideo);

    const wasCurrent = currentVideo && currentVideo.path === oldPath;
    if (wasCurrent) {
        currentVideo = mergedVideo;
        state.currentVideo = currentVideo;
    }

    filterAndRenderGallery(currentTreeSelection);
    updateDirectoryTree();

    if (wasCurrent && viewMode === 'player') {
        selectVideo(mergedVideo);
    }
}

function removeVideoFromState(filePath) {
    if (!videoMap.has(filePath)) {
        return;
    }

    const index = videos.findIndex(v => v.path === filePath);
    if (index !== -1) {
        videos.splice(index, 1);
    }

    videoMap.delete(filePath);

    const wasCurrent = currentVideo && currentVideo.path === filePath;
    if (wasCurrent) {
        closeVideo();
    } else {
        filterAndRenderGallery(currentTreeSelection);
    }

    updateDirectoryTree();
}


// Add videos (legacy support)
document.getElementById('addVideoBtn').addEventListener('click', async () => {
    if (!settings.monitoredDirectories || settings.monitoredDirectories.length === 0) {
        alert('Add at least one monitored directory in Settings before adding videos.');
        return;
    }

    let targetDirectory = currentSelectedDirectory;

    if (!targetDirectory) {
        if (settings.monitoredDirectories.length === 1) {
            const dirObj = settings.monitoredDirectories[0];
            targetDirectory = typeof dirObj === 'string' ? dirObj : dirObj.path;
        } else {
            alert('Select a destination folder from the sidebar before adding videos.');
            return;
        }
    }

    const filePaths = await ipcRenderer.invoke('select-video-files');
    if (!filePaths || filePaths.length === 0) {
        return;
    }

    const copyResults = await Promise.all(filePaths.map(sourcePath => {
        return ipcRenderer.invoke('copy-video-file', { sourcePath, targetDirectory });
    }));

    const failures = copyResults.filter(result => !result || !result.success);
    if (failures.length > 0) {
        const errorMessages = failures
            .map(result => (result && result.error) ? result.error : 'Unknown error')
            .join('\n');
        alert(`Failed to add some videos:\n${errorMessages}`);
    }

    await scanAllDirectories();
});

// Update stats
function updateStats(videosList = videos) {
    const stats = document.getElementById('stats');
    const hasFilters = Boolean(currentSelectedDirectory || searchQuery);
    if (videosList.length === 0) {
        stats.textContent = hasFilters ? 'No videos match current filters' : 'No videos loaded';
        return;
    }

    const totalSize = videosList.reduce((sum, v) => sum + v.size, 0);
    const folderCount = organizeByFolders(videosList).size;
    const displayCount = hasFilters ? `${videosList.length} of ${videos.length}` : videosList.length;
    stats.textContent = `${displayCount} video${videosList.length !== 1 ? 's' : ''} in ${folderCount} folder${folderCount !== 1 ? 's' : ''} • Total: ${formatFileSize(totalSize)}`;
}

// Update directory tree in sidebar
async function updateDirectoryTree() {
    const treeContainer = document.getElementById('folderTree');
    const sidebar = document.getElementById('folderSidebar');
    if (!treeContainer || !sidebar) {
        return;
    }

    const hasDirectories = Array.isArray(settings.monitoredDirectories) && settings.monitoredDirectories.length > 0;
    const isYouTubeConnected = Boolean(youtubeAuthState && youtubeAuthState.status && youtubeAuthState.status.authenticated);
    if (!hasDirectories && !isYouTubeConnected) {
        sidebar.style.display = 'none';
        treeContainer.innerHTML = '';
        return;
    }

    sidebar.style.display = 'block';

    const sections = [];

    // Local directories
    if (hasDirectories) {
        try {
            const structure = await ipcRenderer.invoke('get-directory-structure');
            if (!treeExpandedState.has(LOCAL_ROOT_KEY)) {
                treeExpandedState.set(LOCAL_ROOT_KEY, false);
            }
            const rootExpanded = treeExpandedState.get(LOCAL_ROOT_KEY);
            const rootHasChildren = Array.isArray(structure) && structure.length > 0;
            const rootSelected = currentTreeSelection === LOCAL_ROOT_KEY || (!isYouTubeTreeKey(currentTreeSelection) && currentSelectedDirectory === null);
            const rootExpanderIcon = rootHasChildren
                ? `<span class="tree-expander ${rootExpanded ? 'expanded' : 'collapsed'}">▶</span>`
                : '<span class="tree-expander" style="width: 12px;"></span>';

            let localHtml = `<div class="tree-item root${rootSelected ? ' selected' : ''}" data-path="${LOCAL_ROOT_KEY}" data-has-children="${rootHasChildren}" style="margin-left: 0;">
                ${rootExpanderIcon}
                <span>All Videos</span>
            </div>`;

            const renderDirectoryNode = (node, level = 1) => {
                if (!node || node.isFile) {
                    return '';
                }
                const indent = level * 16;
                const children = Array.isArray(node.children) ? node.children.filter(child => !child.isFile) : [];
                const hasChildren = children.length > 0;
                if (!treeExpandedState.has(node.path)) {
                    treeExpandedState.set(node.path, false);
                }
                const isExpanded = Boolean(treeExpandedState.get(node.path));
                const expanderIcon = hasChildren
                    ? `<span class="tree-expander ${isExpanded ? 'expanded' : 'collapsed'}">▶</span>`
                    : '<span class="tree-expander" style="width: 12px;"></span>';
                const isSelected = currentTreeSelection === node.path;
                let html = `<div class="tree-item${isSelected ? ' selected' : ''}" data-path="${node.path}" data-has-children="${hasChildren}" style="margin-left: ${indent}px;">
                    ${expanderIcon}
                    <span>${node.name}</span>
                </div>`;
                if (hasChildren) {
                    const childrenClass = isExpanded ? 'tree-children expanded' : 'tree-children';
                    html += `<div class="${childrenClass}">`;
                    for (const child of children) {
                        html += renderDirectoryNode(child, level + 1);
                    }
                    html += `</div>`;
                }
                return html;
            };

            if (rootHasChildren) {
                const childrenClass = rootExpanded ? 'tree-children expanded' : 'tree-children';
                localHtml += `<div class="${childrenClass}">`;
                for (const node of structure) {
                    localHtml += renderDirectoryNode(node, 1);
                }
                localHtml += `</div>`;
            }

            sections.push(localHtml);
        } catch (error) {
            logger.error('Error building directory tree:', error);
            sections.push('<p style="color: #A7A9AB; padding: 10px;">Unable to load directories</p>');
        }
    }

    // YouTube content
    if (isYouTubeConnected) {
        if (!treeExpandedState.has(YOUTUBE_ROOT_KEY)) {
            treeExpandedState.set(YOUTUBE_ROOT_KEY, true);
        }
        const youtubeRootExpanded = Boolean(treeExpandedState.get(YOUTUBE_ROOT_KEY));
        const youtubeRootSelected = currentTreeSelection === YOUTUBE_ROOT_KEY;
        const youtubeExpanderIcon = `<span class="tree-expander ${youtubeRootExpanded ? 'expanded' : 'collapsed'}">▶</span>`;
        const { accountId: activeAccountId, account: activeAccount } = getActiveYouTubeAccountRecord();
        const activeAccountLabel = activeAccount ? ` · ${escapeHtml(getYouTubeAccountLabel(activeAccount))}` : '';
        let youtubeHtml = `<div class="tree-item root${youtubeRootSelected ? ' selected' : ''}" data-path="${YOUTUBE_ROOT_KEY}" data-has-children="true" style="margin-left: 0;">
            ${youtubeExpanderIcon}
            <span style="flex: 1;">YouTube${activeAccountLabel}</span>
            <button class="youtube-account-refresh-btn youtube-sidebar-refresh" id="youtubeSidebarRefreshBtn" title="Refresh YouTube library" aria-label="Refresh YouTube library">↻</button>
        </div>`;

        if (youtubeRootExpanded) {
            youtubeHtml += `<div class="tree-children expanded">`;

            if (!activeAccountId) {
                youtubeHtml += `<div class="tree-item" data-path="${YOUTUBE_ROOT_KEY}__empty" data-has-children="false" style="margin-left: 16px; color: #A7A9AB;">
                    <span class="tree-expander" style="width: 12px;"></span>
                    <span>No YouTube account selected</span>
                </div>`;
            } else {
                const encodedAccountId = encodeKeyComponent(activeAccountId);

                YOUTUBE_CATEGORIES.forEach(category => {
                    const categoryKey = buildYouTubeCategoryKey(activeAccountId, category.key);
                    const categorySelected = currentTreeSelection === categoryKey;
                    youtubeHtml += `<div class="tree-item${categorySelected ? ' selected' : ''}" data-path="${categoryKey}" data-has-children="false" style="margin-left: 16px;">
                        <span class="tree-expander" style="width: 12px;"></span>
                        <span>${category.label}</span>
                    </div>`;
                });
            }

            youtubeHtml += `</div>`;
        }

        sections.push(youtubeHtml);
    }

    treeContainer.innerHTML = sections.join('<div style="height: 12px;"></div>');

    // Attach behaviors
    treeContainer.querySelectorAll('.tree-item').forEach(item => {
        item.addEventListener('click', (event) => {
            event.stopPropagation();
            const pathKey = item.dataset.path;
            const hasChildren = item.dataset.hasChildren === 'true';
            const expander = item.querySelector('.tree-expander');
            const isExpanderClick = expander && (event.target === expander || expander.contains(event.target));
            const isRefreshButtonClick = event.target.closest('.youtube-sidebar-refresh');

            // Ignore clicks on the refresh button
            if (isRefreshButtonClick) {
                return;
            }

            if (isExpanderClick && hasChildren) {
                const isExpanded = Boolean(treeExpandedState.get(pathKey));
                treeExpandedState.set(pathKey, !isExpanded);
                if (expander) {
                    expander.classList.toggle('expanded', !isExpanded);
                    expander.classList.toggle('collapsed', isExpanded);
                }
                const childrenContainer = item.nextElementSibling;
                if (childrenContainer) {
                    childrenContainer.classList.toggle('expanded', !isExpanded);
                }
                return;
            }

            treeContainer.querySelectorAll('.tree-item.selected').forEach(node => node.classList.remove('selected'));
            item.classList.add('selected');

            filterAndRenderGallery(pathKey);
        });
    });

    // Attach refresh button behavior
    const sidebarRefreshBtn = document.getElementById('youtubeSidebarRefreshBtn');
    if (sidebarRefreshBtn) {
        let isRefreshing = false;
        sidebarRefreshBtn.addEventListener('click', async (event) => {
            event.stopPropagation();
            if (isRefreshing) {
                return;
            }
            
            const targetAccountId = getResolvedYouTubeAccountId();
            if (!targetAccountId) {
                return;
            }
            
            isRefreshing = true;
            sidebarRefreshBtn.classList.add('loading');
            sidebarRefreshBtn.disabled = true;
            
            try {
                await loadYouTubeLibraryForAccount(targetAccountId, { force: true });
                // Gallery will be updated automatically by loadYouTubeLibraryForAccount
            } catch (error) {
                logger.warn('Failed to refresh YouTube library:', error && error.message ? error.message : error);
            } finally {
                isRefreshing = false;
                sidebarRefreshBtn.classList.remove('loading');
                sidebarRefreshBtn.disabled = false;
            }
        });
    }
}

ipcRenderer.on('audio-scan-progress', (_event, { percent, message }) => {
    if (!editorState || !editorState.elements) return;
    const { highlightDetectProgressFill, highlightDetectProgressText, highlightDetectProgress } = editorState.elements;
    if (highlightDetectProgress) highlightDetectProgress.style.display = 'flex';
    if (highlightDetectProgressFill) highlightDetectProgressFill.style.width = `${Math.min(100, Math.max(0, percent))}%`;
    if (highlightDetectProgressText) highlightDetectProgressText.textContent = message || '';
});

ipcRenderer.on('visual-scan-progress', (_event, { percent, message }) => {
    if (!editorState || !editorState.elements) return;
    const { highlightDetectProgressFill, highlightDetectProgressText, highlightDetectProgress } = editorState.elements;
    if (highlightDetectProgress) highlightDetectProgress.style.display = 'flex';
    if (highlightDetectProgressFill) highlightDetectProgressFill.style.width = `${Math.min(100, Math.max(0, percent))}%`;
    if (highlightDetectProgressText) highlightDetectProgressText.textContent = message || '';
});


// Listen for video file changes
ipcRenderer.on('videos-scanned', (event, data) => {
    const dirName = data.directoryName || data.directory.split(/[\\/]/).pop();
    for (const video of data.videos) {
        if (!videoMap.has(video.path)) {
            video.directoryName = dirName;
            video.directoryPath = data.directory;
            videos.push(video);
            videoMap.set(video.path, video);
        }
    }
    filterAndRenderGallery(currentTreeSelection);
    updateDirectoryTree();
});

ipcRenderer.on('youtube.auth.completed', (_event, payload) => {
    youtubeAuthState.inProgress = false;
    youtubeAuthState.manualAuthUrl = null;

    if (payload && payload.success) {
        youtubeAuthState.status = payload.status || { authenticated: true };
        youtubeAuthState.lastError = null;
        hydrateYouTubeState({ force: true }).catch(error => {
            logger.warn('Failed to refresh YouTube library after authentication:', error);
        });
    } else {
        youtubeAuthState.lastError = payload && payload.error ? payload.error : 'YouTube authentication failed.';
        youtubeLibrary = {
            fetchedAt: 0,
            videos: [],
            playlists: [],
            playlistItems: {},
            errors: youtubeAuthState.lastError
        };
        youtubeState.library = youtubeLibrary;
        state.youtube.library = youtubeLibrary;
        resetYouTubeLibraryIndex();
        if (isYouTubeTreeKey(currentTreeSelection)) {
            filterAndRenderGallery(currentTreeSelection);
        }
    }

    renderYouTubeAuthSection();
});

ipcRenderer.on('youtube.upload.progress', (_event, payload) => {
    handleYouTubeUploadProgress(payload);
});

ipcRenderer.on('youtube.upload.completed', (_event, payload) => {
    handleYouTubeUploadCompleted(payload);
});

ipcRenderer.on('youtube.video.updated', (_event, payload) => {
    if (!payload || !payload.video) {
        return;
    }

    // Get account IDs
    const payloadAccountId = payload.accountId || payload.video.accountId || null;
    const activeAccountId = currentYouTubeAccountId || youtubeActiveAccountId || null;
    
    // Since we now use separate JSON files per account, we should only update
    // if the payload is for the currently active account. If it's for a different
    // account, we'll ignore it and the record will be loaded when that account becomes active.
    if (payloadAccountId != null && activeAccountId != null && payloadAccountId !== activeAccountId) {
        // This update is for a different account - ignore it
        // The record is saved to that account's file and will be loaded when switching accounts
        return;
    }
    
    // If we don't have an active account but have a payload accountId, we can't update
    // (records are account-specific now)
    if (!activeAccountId && payloadAccountId) {
        return;
    }

    // Update local record if we have a videoPath and matching account
    if (payload.videoPath && (payloadAccountId === activeAccountId || !payloadAccountId)) {
        // Pass the accountId to ensure correct account matching
        updateYouTubeRecord(payload.videoPath, {
            ...payload.video,
            lastError: null
        }, payloadAccountId || activeAccountId);

        if (currentVideo && currentVideo.path === payload.videoPath) {
            renderYouTubeDetail(currentVideo);
        }
    }

    // Update YouTube library cache if we have a videoId (for library videos without local path)
    if (payload.video && payload.video.videoId) {
        const accountId = payload.video.accountId || currentYouTubeAccountId || youtubeActiveAccountId || null;
        
        // Helper function to update a library's video
        const updateLibraryVideo = (library) => {
            if (!library || !Array.isArray(library.videos)) {
                return false;
            }
            
            // Ensure we have a valid videoId
            const targetVideoId = payload.video.videoId;
            if (!targetVideoId || typeof targetVideoId !== 'string') {
                logger.warn('Cannot update library video: invalid videoId', payload.video);
                return false;
            }
            
            const videoIndex = library.videos.findIndex(v => v && v.videoId === targetVideoId);
            if (videoIndex >= 0) {
                // Update existing video - preserve ALL existing properties, especially videoId
                const existingVideo = library.videos[videoIndex];
                
                // Ensure videoId is preserved as a string
                const preservedVideoId = String(targetVideoId || existingVideo.videoId || '');
                if (!preservedVideoId) {
                    logger.error('Cannot update library video: no valid videoId found', { existingVideo, payload: payload.video });
                    return false;
                }
                
                library.videos[videoIndex] = {
                    ...existingVideo, // Preserve all existing properties (videoId, thumbnails, url, etc.)
                    videoId: preservedVideoId, // Always use the videoId from payload (it's the source of truth)
                    title: payload.video.title || existingVideo.title,
                    description: payload.video.description !== undefined ? payload.video.description : existingVideo.description,
                    privacyStatus: payload.video.privacy || existingVideo.privacyStatus,
                    embeddable: payload.video.embeddable !== undefined ? payload.video.embeddable : (existingVideo.embeddable !== undefined ? existingVideo.embeddable : true),
                    _localUpdate: true
                };
                return true;
            } else {
                // Video not in library - add it (this can happen if library hasn't loaded yet)
                library.videos.push({
                    videoId: String(targetVideoId),
                    title: payload.video.title || 'YouTube Video',
                    description: payload.video.description || '',
                    privacyStatus: payload.video.privacy || 'unlisted',
                    embeddable: payload.video.embeddable !== undefined ? payload.video.embeddable : true,
                    url: `https://youtu.be/${targetVideoId}`,
                    _localUpdate: true
                });
                return true;
            }
        };
        
        // Update in the current library
        if (youtubeLibrary) {
            updateLibraryVideo(youtubeLibrary);
            resetYouTubeLibraryIndex(youtubeLibrary.accountId || currentYouTubeAccountId || youtubeActiveAccountId || null);
        }
        
        // Also update in the libraryByAccount map to persist across reloads
        if (accountId) {
            if (!youtubeLibraryByAccount.has(accountId)) {
                // Ensure the library exists
                ensureYouTubeLibraryCache(accountId);
            }
            const accountLibrary = youtubeLibraryByAccount.get(accountId);
            if (accountLibrary) {
                updateLibraryVideo(accountLibrary);
            }
        }
        
        // Preserve the videoId in player state if this is the currently playing video
        // This ensures the video remains playable even if the modal was closed
        if (youtubePlayerCurrentVideoId === payload.video.videoId) {
            // The videoId is already set, keep it
        } else if (payload.video.videoId) {
            // If we don't have a current videoId but we just updated one,
            // don't set it here - let the submit handler restore it if needed
        }
    }

    // Don't refresh gallery after metadata update - it would reload the library and potentially lose updates
    // The gallery will refresh naturally when the user navigates or manually refreshes
    // This prevents the library from being reloaded and overwriting our local updates
});

ipcRenderer.on('youtube.video.removed', async (_event, payload) => {
    if (!payload || (!payload.videoPath && !payload.videoId)) {
        return;
    }

    // Handle local video records (has videoPath)
    if (payload.videoPath) {
        removeYouTubeRecord(payload.videoPath);
        youtubeState.jobsByVideoPath.delete(payload.videoPath);

        if (currentVideo && currentVideo.path === payload.videoPath) {
            renderYouTubeDetail(currentVideo);
        }
    }

    // Handle YouTube library videos (videoId only)
    if (payload.videoId && payload.accountId) {
        // Remove from library cache
        const accountId = payload.accountId;
        if (youtubeLibraryByAccount.has(accountId)) {
            const library = youtubeLibraryByAccount.get(accountId);
            if (Array.isArray(library.videos)) {
                library.videos = library.videos.filter(v => v && v.videoId !== payload.videoId);
                // Reset index to rebuild it
                resetYouTubeLibraryIndex(accountId);
                
                // Update current library if it's the active one
                if (youtubeLibrary && youtubeLibrary.accountId === accountId) {
                    youtubeLibrary.videos = library.videos;
                    state.youtube.library = youtubeLibrary;
                }
            }
        }
        
        // If currently viewing YouTube library, refresh the gallery while preserving current view
        if (isYouTubeTreeKey(currentTreeSelection)) {
            // Refresh the library from cache to ensure UI is updated, staying in current folder/view
            filterAndRenderGallery(currentTreeSelection);
            youtubeState.lastGalleryRefresh = Date.now();
        } else if (viewMode === 'gallery') {
            // Not in YouTube view, refresh local gallery
            youtubeState.lastGalleryRefresh = Date.now();
            renderGallery();
        }
    } else if (viewMode === 'gallery' && !isYouTubeTreeKey(currentTreeSelection)) {
        // Only refresh local gallery if we're not in YouTube view
        youtubeState.lastGalleryRefresh = Date.now();
        renderGallery();
    }
});

ipcRenderer.on('video-added', async (event, filePath) => {
    // Rescan the directory to get full video info
    const dirObj = settings.monitoredDirectories.find(d => {
        const dPath = typeof d === 'string' ? d : d.path;
        return filePath.startsWith(dPath);
    });
    if (dirObj) {
        const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
        const dirName = typeof dirObj === 'string' ? dirPath.split(/[\\/]/).pop() : dirObj.name;
        const dirVideos = await ipcRenderer.invoke('scan-directory', dirPath);
        const video = dirVideos.find(v => v.path === filePath);
        if (video && !videoMap.has(video.path)) {
            video.directoryName = dirName;
            video.directoryPath = dirPath;
            videos.push(video);
            videoMap.set(video.path, video);
            filterAndRenderGallery(currentTreeSelection);
            updateDirectoryTree();
        }
    }
});

ipcRenderer.on('video-removed', (event, filePath) => {
    removeVideoFromState(filePath);
});

// Listen for directory structure updates
ipcRenderer.on('directory-structure-updated', () => {
    updateDirectoryTree();
});

// Initialize
const searchInput = document.getElementById('searchInput');

function scheduleSearchAnalytics(query) {
    if (searchAnalyticsTimer) {
        clearTimeout(searchAnalyticsTimer);
    }
    searchAnalyticsTimer = setTimeout(() => {
        const sanitized = (query || '').trim();
        if (sanitized === lastTrackedSearchQuery) {
            return;
        }
        lastTrackedSearchQuery = sanitized;
        trackEvent('search_applied', {
            query_length: sanitized.length,
            is_empty: sanitized.length === 0 ? 1 : 0
        });
    }, 600);
}

searchInput.addEventListener('input', (event) => {
    searchQuery = event.target.value.trim();
    state.searchQuery = searchQuery;
    filterAndRenderGallery(currentTreeSelection);
    scheduleSearchAnalytics(searchQuery);
});

document.addEventListener('keydown', (event) => {
    if (youtubePlayerModal && youtubePlayerModal.classList.contains('active')) {
        if (event.key === 'Escape') {
            event.preventDefault();
            closeYouTubePlayerModal();
            return;
        }
    }

    if (youtubeUploadModal && youtubeUploadModal.classList.contains('active')) {
        if (event.key === 'Escape') {
            event.preventDefault();
            closeYouTubeUploadModal();
            return;
        }

        if (event.key === 'Enter' && (!event.shiftKey && !event.ctrlKey && !event.metaKey)) {
            const targetElement = event.target;
            if (targetElement && youtubeUploadModal.contains(targetElement)) {
                event.preventDefault();
                submitYouTubeUpload();
                return;
            }
        }
    }

    if (saveTemplateModal && saveTemplateModal.classList.contains('active')) {
        if (event.key === 'Escape') {
            event.preventDefault();
            closeSaveTemplateModal();
            return;
        }

        if (event.key === 'Enter' && event.target === saveTemplateInput && !event.shiftKey && !event.ctrlKey && !event.metaKey) {
            event.preventDefault();
            submitSaveTemplate();
            return;
        }
    }

    if (renameModal && renameModal.classList.contains('active')) {
        if (event.key === 'Escape') {
            event.preventDefault();
            closeRenameModal();
            return;
        }

        if (event.key === 'Enter' && event.target === renameInput && !event.shiftKey && !event.ctrlKey && !event.metaKey) {
            event.preventDefault();
            submitRename();
            return;
        }
    }

    if (event.key === 'Escape' && viewMode === 'player') {
        event.preventDefault();
        closeVideo();
    }
    if (event.key === 'Escape' && viewMode === 'editor') {
        event.preventDefault();
        closeVideo();
        return;
    }

    if (viewMode === 'editor' && !event.shiftKey && !event.metaKey && !event.ctrlKey && !event.altKey) {
        const tagName = (event.target && event.target.tagName) ? event.target.tagName.toLowerCase() : '';
        if (tagName !== 'input' && tagName !== 'textarea' && tagName !== 'button') {
            if (event.key === 's' || event.key === 'S' || event.key === 'b' || event.key === 'B') {
                event.preventDefault();
                addEditorSplit();
            }
        }
    }

    // Global spacebar hotkey to play/pause video (unless in input field)
    if (event.code === 'Space' || event.key === ' ') {
        const targetTagName = (event.target && event.target.tagName) ? event.target.tagName.toLowerCase() : '';
        const isInputField = targetTagName === 'input' || targetTagName === 'textarea' || 
            (event.target && event.target.isContentEditable);
        
        if (!isInputField && (viewMode === 'player' || viewMode === 'editor')) {
            event.preventDefault();
            const playerContainer = document.getElementById('playerContainer');
            const videoElement = playerContainer ? playerContainer.querySelector('video') : null;
            
            if (videoElement) {
                if (videoElement.paused) {
                    videoElement.play().catch(() => {});
                } else {
                    videoElement.pause();
                }
            }
        }
    }

    // Frame-by-frame navigation with arrow keys
    if ((viewMode === 'player' || viewMode === 'editor') && !event.shiftKey && !event.metaKey && !event.ctrlKey && !event.altKey) {
        const targetTagName = (event.target && event.target.tagName) ? event.target.tagName.toLowerCase() : '';
        const isInputField = targetTagName === 'input' || targetTagName === 'textarea' || 
            (event.target && event.target.isContentEditable);
        
        if (!isInputField) {
            const playerContainer = document.getElementById('playerContainer');
            const videoElement = playerContainer ? playerContainer.querySelector('video') : null;
            
            if (videoElement && (event.key === 'ArrowLeft' || event.key === 'ArrowRight')) {
                event.preventDefault();
                
                // Calculate frame step based on mode
                // Player mode: 1 frame, Editor mode: 1 frame for marker movement
                const frameStep = SINGLE_FRAME_DURATION;
                
                if (event.key === 'ArrowLeft') {
                    // Skip exactly 1 frame (1/30 second)
                    videoElement.currentTime = Math.max(0, videoElement.currentTime - frameStep);
                    
                    // In editor mode, also move the template marker
                    if (viewMode === 'editor' && editorState) {
                        const duration = editorState.duration || 0;
                        const markerTime = Math.max(0, (editorState.templateMarker || 0) - frameStep);
                        editorState.templateMarker = markerTime;
                        // Update the UI for the marker
                        if (typeof updateHighlightDetectUI === 'function') {
                            updateHighlightDetectUI();
                        }
                    }
                    
                    // Start delayed continuous skipping if not already scheduled
                    if (frameSkipDirection !== 'left') {
                        clearInterval(frameSkipInterval);
                        clearTimeout(frameSkipTimeout);
                        frameSkipDirection = 'left';
                        
                        // Start continuous skipping after 1 second hold
                        frameSkipTimeout = setTimeout(() => {
                            frameSkipInterval = setInterval(() => {
                                if (videoElement && frameSkipDirection === 'left') {
                                    videoElement.currentTime = Math.max(0, videoElement.currentTime - (1 / 5)); // 5 fps
                                    
                                    // Also move marker continuously in editor mode
                                    if (viewMode === 'editor' && editorState) {
                                        const duration = editorState.duration || 0;
                                        const markerTime = Math.max(0, (editorState.templateMarker || 0) - (1 / 5));
                                        editorState.templateMarker = markerTime;
                                        if (typeof updateHighlightDetectUI === 'function') {
                                            updateHighlightDetectUI();
                                        }
                                    }
                                }
                            }, FRAME_SKIP_INTERVAL_MS);
                        }, FRAME_SKIP_DELAY_MS);
                    }
                } else if (event.key === 'ArrowRight') {
                    // Skip exactly 1 frame (1/30 second)
                    videoElement.currentTime = Math.min(videoElement.duration, videoElement.currentTime + frameStep);
                    
                    // In editor mode, also move the template marker
                    if (viewMode === 'editor' && editorState) {
                        const duration = editorState.duration || 0;
                        const markerTime = Math.min(duration, (editorState.templateMarker || 0) + frameStep);
                        editorState.templateMarker = markerTime;
                        // Update the UI for the marker
                        if (typeof updateHighlightDetectUI === 'function') {
                            updateHighlightDetectUI();
                        }
                    }
                    
                    // Start delayed continuous skipping if not already scheduled
                    if (frameSkipDirection !== 'right') {
                        clearInterval(frameSkipInterval);
                        clearTimeout(frameSkipTimeout);
                        frameSkipDirection = 'right';
                        
                        // Start continuous skipping after 1 second hold
                        frameSkipTimeout = setTimeout(() => {
                            frameSkipInterval = setInterval(() => {
                                if (videoElement && frameSkipDirection === 'right') {
                                    videoElement.currentTime = Math.min(videoElement.duration, videoElement.currentTime + (1 / 5)); // 5 fps
                                    
                                    // Also move marker continuously in editor mode
                                    if (viewMode === 'editor' && editorState) {
                                        const duration = editorState.duration || 0;
                                        const markerTime = Math.min(duration, (editorState.templateMarker || 0) + (1 / 5));
                                        editorState.templateMarker = markerTime;
                                        if (typeof updateHighlightDetectUI === 'function') {
                                            updateHighlightDetectUI();
                                        }
                                    }
                                }
                            }, FRAME_SKIP_INTERVAL_MS);
                        }, FRAME_SKIP_DELAY_MS);
                    }
                }
            }
        }
    }
});

// Stop continuous frame skipping on keyup
document.addEventListener('keyup', (event) => {
    if (event.key === 'ArrowLeft' || event.key === 'ArrowRight') {
        clearInterval(frameSkipInterval);
        clearTimeout(frameSkipTimeout);
        frameSkipInterval = null;
        frameSkipTimeout = null;
        frameSkipDirection = null;
    }
});

// Stop frame skipping when window loses focus
window.addEventListener('blur', () => {
    clearInterval(frameSkipInterval);
    clearTimeout(frameSkipTimeout);
    frameSkipInterval = null;
    frameSkipTimeout = null;
    frameSkipDirection = null;
});

// Sidebar toggle for responsive layout
const sidebarToggle = document.getElementById('sidebarToggle');
const sidebar = document.getElementById('folderSidebar');
if (sidebarToggle && sidebar) {
    sidebarToggle.addEventListener('click', () => {
        sidebar.classList.toggle('open');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', (event) => {
        if (window.innerWidth <= 900 && 
            sidebar.classList.contains('open') &&
            !sidebar.contains(event.target) &&
            !sidebarToggle.contains(event.target)) {
            sidebar.classList.remove('open');
        }
    });
}

filterAndRenderGallery(currentTreeSelection);

loadSettings().then(() => {
    if (settings.monitoredDirectories.length > 0) {
        scanAllDirectories();
    } else {
        updateDirectoryTree();
    }
});

// Video file info modal
async function showVideoInfoModal() {
    if (!editorState || !editorState.video || !editorState.video.path) {
        setEditorStatus('No video loaded.', 'info');
        return;
    }

    let modal = document.getElementById('videoInfoModal');
    if (!modal) {
        modal = document.createElement('div');
        modal.id = 'videoInfoModal';
        modal.className = 'modal';
        modal.innerHTML = `
            <div class="modal-content modal-medium">
                <div class="modal-header">
                    <h3>Video File Info</h3>
                    <button class="modal-close" id="closeVideoInfoModal">&times;</button>
                </div>
                <div class="modal-body" id="videoInfoBody" style="max-height: 60vh; overflow-y: auto;">
                    <div style="color: #888;">Loading...</div>
                </div>
            </div>
        `;
        document.body.appendChild(modal);

        document.getElementById('closeVideoInfoModal').addEventListener('click', () => {
            modal.classList.remove('active');
        });

        modal.addEventListener('click', (e) => {
            if (e.target === modal) modal.classList.remove('active');
        });
    }

    modal.classList.add('active');
    const bodyEl = document.getElementById('videoInfoBody');
    bodyEl.innerHTML = '<div style="color: #888;">Loading...</div>';

    try {
        const result = await ipcRenderer.invoke('get-video-metadata', editorState.video.path);
        if (!result || !result.success) {
            bodyEl.innerHTML = `<div style="color: #c44;">${result && result.error ? result.error : 'Failed to load metadata.'}</div>`;
            return;
        }

        const m = result.metadata;
        const fmt = m.format || {};
        const vid = m.video || {};
        const aud = m.audio || {};

        let html = `
            <table class="video-info-table" style="width: 100%; border-collapse: collapse; font-size: 14px;">
                <tr><td style="padding: 6px 12px 6px 0; color: #888; vertical-align: top; width: 140px;">File</td><td style="padding: 6px 0;">${escapeHtml(m.name || '—')}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Format</td><td style="padding: 6px 0;">${escapeHtml(fmt.longName || fmt.name || '—')}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Duration</td><td style="padding: 6px 0;">${fmt.durationFormatted || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Size</td><td style="padding: 6px 0;">${fmt.sizeFormatted || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Bitrate</td><td style="padding: 6px 0;">${fmt.bitrateFormatted || '—'}</td></tr>
                <tr><td colspan="2" style="padding: 12px 0 6px; border-top: 1px solid #2a2a2a;"><strong style="color: #fff;">Video</strong></td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Resolution</td><td style="padding: 6px 0;">${vid.resolution || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Codec</td><td style="padding: 6px 0;">${escapeHtml(vid.codecLongName || vid.codec || '—')}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">FPS</td><td style="padding: 6px 0;">${vid.fps !== '—' ? vid.fps : '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Pixel format</td><td style="padding: 6px 0;">${vid.pixelFormat || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Aspect ratio</td><td style="padding: 6px 0;">${vid.aspectRatio || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Video bitrate</td><td style="padding: 6px 0;">${vid.bitrateFormatted || '—'}</td></tr>
        `;

        if (aud && (aud.codec || aud.codecLongName)) {
            html += `
                <tr><td colspan="2" style="padding: 12px 0 6px; border-top: 1px solid #2a2a2a;"><strong style="color: #fff;">Audio</strong></td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Codec</td><td style="padding: 6px 0;">${escapeHtml(aud.codecLongName || aud.codec || '—')}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Sample rate</td><td style="padding: 6px 0;">${aud.sampleRate || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Channels</td><td style="padding: 6px 0;">${aud.channels || '—'}</td></tr>
                <tr><td style="padding: 6px 12px 6px 0; color: #888;">Audio bitrate</td><td style="padding: 6px 0;">${aud.bitrateFormatted || '—'}</td></tr>
            `;
        }

        html += '</table>';
        bodyEl.innerHTML = html;
    } catch (err) {
        bodyEl.innerHTML = `<div style="color: #c44;">${escapeHtml(err && err.message ? err.message : 'Failed to load metadata.')}</div>`;
    }
}

function escapeHtml(str) {
    if (str == null || str === '') return '';
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
}

// Visual scan results modal
function showVisualResultsModal() {
    const results = editorState.lastVisualScanResults;
    if (!results || results.length === 0) {
        setEditorStatus('No scan results available. Run a visual scan first.', 'info');
        return;
    }
    
    let modal = document.getElementById('visualResultsModal');
    if (!modal) {
        modal = document.createElement('div');
        modal.id = 'visualResultsModal';
        modal.className = 'modal';
        modal.innerHTML = `
            <div class="modal-content" style="max-width: 90%; max-height: 90%; width: 900px;">
                <div class="modal-header">
                    <h3>Visual Scan Results</h3>
                    <button class="modal-close" id="closeVisualResultsModal">&times;</button>
                </div>
                <div class="modal-body" id="visualResultsBody" style="max-height: 70vh; overflow-y: auto;"></div>
            </div>
        `;
        document.body.appendChild(modal);
        
        document.getElementById('closeVisualResultsModal').addEventListener('click', () => {
            modal.style.display = 'none';
        });
        
        modal.addEventListener('click', (e) => {
            if (e.target === modal) modal.style.display = 'none';
        });
    }
    
    const threshold = editorState.visualThreshold || 0.7;
    const matches = results.filter(r => r.similarity >= threshold).sort((a, b) => b.similarity - a.similarity);
    const nonMatches = results.filter(r => r.similarity < threshold).sort((a, b) => b.similarity - a.similarity);
    
    let html = `
        <div style="margin-bottom: 15px;">
            <strong>Found ${matches.length} matches</strong> (threshold: ${threshold})
            <span style="color: #888; margin-left: 15px;">Total frames: ${results.length}</span>
        </div>
    `;
    
    if (matches.length > 0) {
        html += '<h4 style="margin: 15px 0 10px;">Matches</h4><div class="visual-results-grid">';
        matches.forEach(m => {
            const timeStr = formatTimestamp(m.timestamp, 2);
            const imgTag = m.thumbnail ? `<img src="data:image/png;base64,${m.thumbnail}" style="max-width: 100px; max-height: 64px; background: #000; border-radius: 2px; margin-bottom: 4px;">` : '';
            html += `
                <div class="visual-result-item" data-time="${m.timestamp}" style="cursor: pointer; border: 2px solid #4CAF50; border-radius: 4px; padding: 8px; margin: 4px; display: inline-block; text-align: center; min-width: 80px;">
                    ${imgTag}
                    <div style="font-size: 12px; color: #888;">${timeStr}</div>
                    <div style="font-size: 16px; font-weight: bold; color: #4CAF50;">${(m.similarity * 100).toFixed(1)}%</div>
                </div>
            `;
        });
        html += '</div>';
    }
    
    if (nonMatches.length > 0 && matches.length < 50) {
        const showCount = Math.min(100, nonMatches.length);
        html += `<h4 style="margin: 15px 0 10px;">Top Non-Matches (showing ${showCount})</h4><div class="visual-results-grid">`;
        for (let i = 0; i < showCount; i++) {
            const m = nonMatches[i];
            const timeStr = formatTimestamp(m.timestamp, 2);
            const imgTag = m.thumbnail ? `<img src="data:image/png;base64,${m.thumbnail}" style="max-width: 80px; max-height: 48px; background: #000; border-radius: 2px; margin-bottom: 4px; opacity: 0.7;">` : '';
            html += `
                <div class="visual-result-item" data-time="${m.timestamp}" style="cursor: pointer; border: 1px solid #555; border-radius: 4px; padding: 8px; margin: 4px; display: inline-block; text-align: center; min-width: 70px; opacity: 0.7;">
                    ${imgTag}
                    <div style="font-size: 12px; color: #888;">${timeStr}</div>
                    <div style="font-size: 14px; color: #aaa;">${(m.similarity * 100).toFixed(1)}%</div>
                </div>
            `;
        }
        html += '</div>';
    }
    
    document.getElementById('visualResultsBody').innerHTML = html;
    
    modal.querySelectorAll('.visual-result-item').forEach(item => {
        item.addEventListener('click', () => {
            const time = parseFloat(item.dataset.time);
            if (editorState.video && editorState.video.path) {
                seekEditorTo(time);
            }
            modal.style.display = 'none';
        });
    });
    
    modal.style.display = 'block';
}

// refreshYouTubeAuthStatus will call hydrateYouTubeState if user is authenticated and we request it
refreshYouTubeAuthStatus({ hydrateIfAuthenticated: true });
renderYouTubeAuthSection();

// Show gallery by default
document.getElementById('galleryContainer').style.display = 'block';

initializeAnalytics().then(() => {
    trackScreenView(viewMode || 'gallery', { trigger: 'app_start', force: true });
}).catch(() => {});

